#' Direct regression method for continuous response
#'
#' @param data data
#' @param ID id of patient
#' @param true_condition true_condition of patient,binary
#' @param occasion method,1,2,...,Q
#' @param test_result result of test
#' @param X covariate of patient
#' @param func_H link function
#' @param initial_guess the initial guess of the equation
#' @param n_T number of cutting point of empirical function
#'
#'@import nleqslv
#'@import BB
#'@import reshape2
#'@import dplyr
#'@import ggplot2
#'@import haven
#' @return solution of parameter, two figures of ROC,AUC
#' @export
#'
Cont_ROC<-function(data,ID,true_condition,occasion,test_result,X,func_H,initial_guess,n_T)
{
  Q=length(table(data[[occasion]]))#Q is the number of occasions
  N=length(data[[ID]])/Q
  n_1=nrow(data[data[[true_condition]]==1,])/Q
  n_0=N-n_1
  set_gamma=seq(1/n_T,(n_T-1)/n_T,1/n_T)
  P=length(set_gamma)
  U=array(0,dim=c(n_1,Q,P))
  set_not_patient=matrix(0,nrow=n_0,ncol=Q)
  id_n_0=unique(data[data[[true_condition]]==0,][[ID]])
  id_n_1=unique(data[data[[true_condition]]==1,][[ID]])
  for(q in 1:Q)
  {
    set_not_patient[,q]=data[data[[true_condition]]==0&data[[occasion]]==q,][[test_result]]
  }
  F_inv<-function(threshold,q)
  {
    set=sort(set_not_patient[,q])
    return(set[ceiling(n_0*(1-threshold))])
  }
  for(i in 1:n_1)
    for(q in 1:Q)
      for(p in 1:P)
      {
        U[i,q,p]=as.numeric(I(data[data[[ID]]==id_n_1[i]&data[[occasion]]==q,][[test_result]]>F_inv(set_gamma[p],q)))
      }
  x=rep(0,N)#derive the covariate vector x
  for(i in 1:N)
  {
    x[i]=subset(data,data[[ID]]==i)[[X]][1]
  }
  solve_my_eq<-function(parameter)
  {
    #setting the parameters
    gamma=matrix(0,nrow=Q,ncol=2)
    beta=matrix(0,nrow=Q,ncol=2)
    for(q in 1:Q)
    {
      gamma[q,]=parameter[(4*(q-1)+1):(4*(q-1)+2)]
      beta[q,]=parameter[(4*(q-1)+3):(4*(q-1)+4)]
    }
    yita=array(0,dim=c(n_1,Q,P))
    W=array(0,dim=c(n_1,Q,P,4))
    S=matrix(0,4*Q,1)
    for(j in 1:n_1)
      for(q in 1:Q)
        for(p in 1:P)
        { i=id_n_1[j]
        temp=set_gamma[p]
        yita[j,q,p]=pnorm(gamma[q,1]+gamma[q,2]*qnorm(temp)+beta[q,1]*x[i]+beta[q,2]*x[i]*qnorm(temp))
        W[j,q,p,]=dnorm(gamma[q,1]+gamma[q,2]*qnorm(temp)+beta[q,1]*x[i]+beta[q,2]*x[i]*qnorm(temp))/(yita[j,q,p]*(1-yita[j,q,p]))*c(1,qnorm(temp),x[i],x[i]*qnorm(temp))
        W_temp=matrix(0,4*Q,1)
        W_temp[(4*(q-1)+1):(4*(q-1)+4),]=W[j,q,p,]
        S=S+W_temp%*%(U[j,q,p]-yita[j,q,p])
        }
    return(S)
  }
  #estimating the parameters
  solution=nleqslv(initial_guess,solve_my_eq)
  parameter=solution[["x"]]
  gamma=matrix(0,nrow=Q,ncol=2)
  beta=matrix(0,nrow=Q,ncol=2)
  for(q in 1:Q)
  {
    gamma[q,]=parameter[(4*(q-1)+1):(4*(q-1)+2)]
    beta[q,]=parameter[(4*(q-1)+3):(4*(q-1)+4)]
  }
  #plotting the ROC curve
  seq=seq(0.01,0.99,by=0.01)
  Cont_ROC_curve<-function(q,x)#derive the ROC curve
  {
    Cont_ROC_func<-function(p)
    {
      pnorm(gamma[q,1]+gamma[q,2]*qnorm(p)+beta[q,1]*x+beta[q,2]*x*qnorm(p))
    }
    auc=integrate(Cont_ROC_func,0.00001,0.99999)
    return(list(sapply(seq,Cont_ROC_func),auc$value))
  }
  outcome_0=seq
  outcome_1=seq
  AUC=0
    for(q in 1:Q)
    {
      outcome_0=cbind(outcome_0,Cont_ROC_curve(q,0)[[1]])
      outcome_1=cbind(outcome_1,Cont_ROC_curve(q,1)[[1]])
    for(x in 0:1)
      AUC=cbind(AUC,Cont_ROC_curve(q,x)[[2]])
    }
  AUC=as.numeric(AUC[-1])
  outcome_0=as.data.frame(outcome_0)
  colnames(outcome_0)=c("x",letters[1:Q])
  outcome_1=as.data.frame(outcome_1)
  colnames(outcome_1)=c("x",letters[1:Q])
  outcome_0=melt(outcome_0,id.vars=c("x"),variable.name="variable",value.name = "value")
  #print(outcome_0)
  outcome_1=melt(outcome_1,id.vars=c("x"),variable.name="variable",value.name = "value")
  fig0=ggplot(outcome_0,aes(x=x,y=value,color=variable))+geom_point()
  fig1=ggplot(outcome_1,aes(x=x,y=value,color=variable))+geom_point()
  return(list(solution$x,fig0,fig1,AUC))
}
