#' @title Indirect Regression Model for Ordinal-Scale ROC Curves
#'
#' @description you can use this function to fit an indirect regression model for
#'     ordinal-scale ROC curves. This function will print and return the parameters
#'     of the model and the AUC of the corresponding ROC curve, and will make an
#'     image of the ROC curve and its confidence band.
#'
#' @param location a two-sided formula with the response variable on the LHS and
#'     the predictor for the location on the RHS.
#' @param scale a one-sided formula with the predictor for the scale on the RHS.
#' @param data a data frame in which to evaluate the indirect regression model.
#' @param newdata a data frame for generating ROC curve.
#' @param group a character representing the name of the column in data that
#'     represents the true condition of subjects.
#' @param link link function, i.e., the type of location-scale distribution assumed
#'     for the latent distribution. "probit"(default), "logit" or "cauchit".
#' @param level the confidence level. Default 0.95.
#' @param n_boot the number of bootstrap samples to draw. Default 1000.
#'
#' @import ordinal
#' @import stats
#' @import graphics
#'
#' @return a list of the parameters of the model and the AUC of the corresponding ROC curve.
#'
#' @examples
#' \donttest{
#' data("data_ordinal")
#' oiROC(location = t ~ d * (x1 + x2),
#'       scale = ~ d * (x1 + x2),
#'       data = data_ordinal,
#'       newdata = data.frame(x1 = c(0), x2 = c(0)),
#'       group = "d")
#' }
#'
#' @export

oiROC <- function(location, scale, data, newdata, group, link = "probit",
                  level = 0.95, n_boot = 1000){

  f_link <- make.link(link)
  response <- all.vars(location)[1]
  data[[response]] <- factor(data[[response]])

  # Generate new data for calculating ROC curve
  newdata_0 <- newdata
  newdata_0[[group]] <- 0
  newdata_1 <- newdata
  newdata_1[[group]] <- 1

  if (length(attr(terms(location), "term.labels")) == 0){
    loc_tmp <- as.formula(~ -1)
  }
  else{
    loc_tmp <- reformulate(attr(terms(location), "term.labels"), intercept = FALSE)
  }

  if (length(attr(terms(scale), "term.labels")) == 0){
    sca_tmp <- as.formula(~ -1)
  }
  else{
    sca_tmp <- reformulate(attr(terms(scale), "term.labels"), intercept = FALSE)
  }

  # Generate Design Matrix
  data_loc <- model.matrix(loc_tmp, data)
  data_sca <- model.matrix(sca_tmp, data)

  newdata_loc_0 <- model.matrix(loc_tmp, newdata_0)
  newdata_loc_1 <- model.matrix(loc_tmp, newdata_1)
  newdata_sca_0 <- model.matrix(sca_tmp, newdata_0)
  newdata_sca_1 <- model.matrix(sca_tmp, newdata_1)

  # Calculate the value of the ROC curve
  cal_ROC <- function(fpr, beta, alpha){
    a <- sum((newdata_loc_1 - newdata_loc_0) * beta) / exp(sum(newdata_sca_1 * alpha))
    b <- exp(sum(newdata_sca_0 * alpha)) / exp(sum(newdata_sca_1 * alpha))

    return(1 - f_link$linkinv(b * f_link$linkfun(1 - fpr) - a))
  }

  # Fit an ordinal regression
  model <- clm(formula = location, scale = scale, data = data, link = link)

  ts <- model$alpha
  beta <- model$beta
  alpha <- model$zeta

  n0 <- length(ts)
  n1 <- length(beta) + n0
  n2 <- length(alpha) + n1

  # Calculation of the information matrix, used to calculate the standard deviation of the parameter
  info_mat <- function(ts, beta, alpha){
    mat <- matrix(0, nrow = n2, ncol = n2)

    c_w <- c(-Inf, ts, Inf)
    h_loc <- rep(0.0, nrow(data))
    h_scale <- rep(1.0, nrow(data))

    if ((n1 - n0) > 0){
      h_loc <- as.vector(data_loc %*% beta)
    }
    if ((n2 - n1) > 0){
      h_scale <- as.vector(exp(data_sca %*% alpha))
    }

    m <- (tcrossprod(rep(1, length(h_loc)), c_w) - h_loc) / h_scale
    dm <- f_link$mu.eta(m)

    m_0 <- m[, 1: (n0 + 1)]
    m_1 <- m[, 2: (n0 + 2)]
    pm <- f_link$linkinv(m_1) - f_link$linkinv(m_0)
    dm_0 <- dm[, 1: (n0 + 1)]
    dm_1 <- dm[, 2: (n0 + 2)]

    m_0[, 1] <- 0
    m_1[, (n0 + 1)] <- 0

    if (n2 - n1> 0){
      tmp_A <- (dm_0 * m_0 - dm_1 * m_1) / pm
      mat[(n1 + 1): n2, (n1 + 1): n2] <- crossprod(data_sca, diag(rowSums(tmp_A^2 * pm)) %*% data_sca)

      if (n1 - n0 > 0){
        mat[(n1 + 1): n2, (n0 + 1): n1] <- crossprod(data_sca,
          diag(rowSums(tmp_A * (dm_0 - dm_1) / h_scale)) %*% data_loc)
        mat[(n0 + 1): n1, (n1 + 1): n2] <- t(mat[(n1 + 1): n2, (n0 + 1): n1])
      }

      if (n0 > 0){
        mat[(n1 + 1): n2, 1: n0] <- crossprod(data_sca,
          ((tmp_A[, 1: n0] - tmp_A[, 2: (n0 + 1)]) * dm_1[, 1: n0] / h_scale))
        mat[1: n0, (n1 + 1): n2] <- t(mat[(n1 + 1): n2, 1: n0])
      }
    }

    dm_0 <- dm_0 / h_scale
    dm_1 <- dm_1 / h_scale

    if (n1 - n0 > 0){
      tmp_B <- (dm_0 - dm_1) / pm
      mat[(n0 + 1): n1, (n0 + 1): n1] <- crossprod(data_loc,
        diag((rowSums(tmp_B^2 * pm))) %*% data_loc)

      if (n0 > 0){
        mat[(n0 + 1): n1, 1: n0] <- crossprod(data_loc,
          ((tmp_B[, 1: n0] - tmp_B[, 2: (n0 + 1)]) * dm_1[, 1: n0]))
        mat[1: n0, (n0 + 1): n1] <- t(mat[(n0 + 1): n1, 1: n0])
      }
    }

    if (n0 > 0){
      for (i in 1: n0){
        mat[i, i] <- sum((dm_1[, i])^2 * (1 / pm[, i] + 1 / pm[, (i + 1)]))
      }
      if (n0 > 1){
        for (i in 1: (n0 - 1)){
          mat[i, (i + 1)] <- -sum(dm_1[, i] * dm_1[, (i + 1)] / pm[, (i + 1)])
          mat[(i + 1), i] <- mat[i, (i + 1)]
        }
      }
    }

    return(mat)
  }

  std <- sqrt(diag(solve(info_mat(ts, beta, alpha))))

  # Generate fpr_plot for mapping, calculate se_plot and AUC
  fpr_plot <- seq(0, 1, 0.001)
  se_plot <- cal_ROC(fpr_plot, beta, alpha)
  AUC <- integrate(cal_ROC, 0.0, 1.0, beta, alpha)$value

  # Use bootstrap to calculate parameters, standard deviation of ROC curves and AUCs
  se_std <- matrix(nrow = length(se_plot), ncol = 0)
  AUC_std <- c()

  for (iter in 1: n_boot){
    stop <- FALSE
    # Prevent errors from being reported due to too many duplicate samples
    while (!stop){
      index <- sample(rownames(data), nrow(data), replace = TRUE)

      result <- tryCatch({
        model_boot <- clm(formula = location, scale = scale, data = data[index, ], link = link)
        1
      }, warning = function(w){
        2
      }, error = function(e){
        3
      }, finally = {})

      if (result == 1){
        stop <- TRUE
      }
    }

    se_boot <- cal_ROC(fpr_plot, model_boot$beta, model_boot$zeta)
    se_std <- cbind(se_std, se_boot)
    AUC_std <- c(AUC_std, integrate(cal_ROC, 0.0, 1.0, model_boot$beta, model_boot$zeta)$value)
  }

  se_std <- sqrt(apply(se_std, MARGIN = 1, var))
  AUC_std <- sqrt(var(AUC_std))

  # Print the estimates of the parameters along with the standard deviation
  if (n0 > 0){
    ts_std <- std[1: n0]
    ts_info <- cbind(ts, ts_std)
    colnames(ts_info) <- c("Estimate", "std. Error")
    cat("Threshold coefficients: \n")
    print(ts_info, digits = 4)
  }

  if (n1 - n0 > 0){
    loc_std <- std[(n0 + 1): n1]
    loc_info <- cbind(beta, loc_std)
    colnames(loc_info) <- c("Estimate", "std. Error")
    cat("\n")
    cat("Location coefficients: \n")
    print(loc_info, digits = 4)
  }

  if (n2 - n1 > 0){
    scale_std <- std[(n1 + 1): n2]
    scale_info <- cbind(alpha, scale_std)
    colnames(scale_info) <- c("Estimate", "std. Error")
    cat("\n")
    cat("Scale coefficients (log link): \n")
    print(scale_info, digits = 4)
  }

  AUC_info <- cbind(AUC, AUC_std)
  colnames(AUC_info) <- c("Estimate", "std. Error")
  rownames(AUC_info) <- c("AUC")
  cat("\n")
  cat("AUC: \n")
  print(AUC_info, digits = 4)

  # plot
  plot(se_plot ~ fpr_plot, type = "l", main = "ROC Curve Using Indirect Regression Model",
       xlim = c(0.0, 1.0), ylim = c(0.0, 1.0), xlab = "FPR", ylab = "TPR")
  lines(se_plot - qnorm((1 + level) / 2) * se_std ~ fpr_plot, lty = 2)
  lines(se_plot + qnorm((1 + level) / 2) * se_std ~ fpr_plot, lty = 2)
  legend("bottomright", legend = c("ROC Curve", "CI"), lty = c(1, 2))

  list_return <- list(threshold = ts, location = beta, scale = alpha, AUC = AUC)
}
