#' @title Direct Parametric Regression Model for Ordinal-Scale ROC Curves
#'
#' @description you can use this function to fit an direct parametric regression
#'     model for ordinal-scale ROC curves. This function will print and return
#'     the parameters of the model and the AUC of the corresponding ROC curve,
#'     and will make an image of the ROC curve and its confidence band.
#'
#' @param vars a vector of characters representing the covariates in the model.
#' @param eta_vars a vector of characters representing the covariates interacting
#'     with functions of p in the model.
#' @param data a data frame consisting of data from subjects without condition.
#' @param data_d a data frame consisting of data from subjects with condition.
#' @param newdata_d a data frame for generating ROC curve.
#' @param response a character representing the name of the column in both data
#'     and data_d that represents the test results of subjects.
#' @param link link function in direct regression model. "probit", "logit" or "cauchit".
#'     Default "probit".
#' @param base base functions in direct regression model. A vector consisting of
#'     one or more of "probit", "logit" and "cauchit".
#' @param Intercept indicates whether the model contains a constant term. Default TRUE.
#' @param eta function interacting with eta_vars. "probit", "logit" or "cauchit".
#'     Default "probit".
#' @param link_ordinal link function of the ordinal regression model of the survival
#'     function of subjects' latent variable without condition. "probit",
#'     "logit" or "cauchit". Default "probit".
#' @param level the confidence level. Default 0.95.
#' @param n_boot the number of bootstrap samples to draw. Default 1000.
#'
#' @import ordinal
#' @import nleqslv
#' @import stats
#' @import graphics
#'
#' @return a list of the parameters of the model and the AUC of the corresponding ROC curve.
#'
#' @examples
#' \donttest{
#' data("data_ordinal")
#' data <- data_ordinal[data_ordinal[["d"]] == 0, c("t", "x1", "x2")]
#' data_d <- data_ordinal[data_ordinal[["d"]] == 1, c("t", "x1", "x2")]
#' odpROC(vars = c("x1", "x2"),
#'        eta_vars = c(),
#'        data = data,
#'        data_d = data_d,
#'        newdata_d = data.frame(x1 = c(0), x2 = c(0)),
#'        response = "t")
#' }
#'
#' @export

odpROC <- function(vars, eta_vars, data, data_d, newdata_d, response,
                   link = "probit", base = c("probit"), Intercept = TRUE, eta = "probit",
                   link_ordinal = "probit", level = 0.95, n_boot = 1000){

  base <- unique(base)

  # Calculate the value of the ROC curve
  cal_ROC <- function(p, beta, gamma, alpha){
    W_new <- sum(beta * newdata_d[vars])
    W_eta_new <- sum(gamma * newdata_d[eta_vars])

    vec_eta_new <- make.link(eta)$linkfun(p)
    mat_base_new <- matrix(nrow = length(p), ncol = 0)
    if (Intercept){
      mat_base_new <- cbind(mat_base_new, rep(1, length(p)))
    }
    for (bs in base){
      mat_base_new <- cbind(mat_base_new, make.link(bs)$linkfun(p))
    }

    return(make.link(link)$linkinv(W_new + W_eta_new * vec_eta_new +
                        as.vector(mat_base_new %*% alpha)))
  }

  # Use auxiliary function odpROC_ to get parameter estimates
  par_list <- odpROC_(vars, eta_vars, data, data_d, response, link, base,
                      Intercept, eta, link_ordinal)
  beta <- par_list$beta
  gamma <- par_list$gamma
  alpha <- par_list$alpha

  if (length(beta) != 0){
    names(beta) <- vars
  }
  if (length(gamma) != 0){
    names(gamma) <- paste(eta_vars, eta, sep = ":")
  }

  if (Intercept){
    names(alpha) <- c("(Intercept)", base)
  }
  else {
    if (length(base) != 0){
      names(alpha) <- base
    }
  }

  # Generate fpr_plot for mapping, calculate se_plot and AUC
  fpr_plot <- seq(0.001, 0.999, 0.001)
  se_plot <- cal_ROC(fpr_plot, beta, gamma, alpha)
  AUC <- integrate(cal_ROC, 0.0, 1.0, beta, gamma, alpha)$value

  # Use bootstrap to calculate parameters, standard deviation of ROC curves and AUCs
  beta_boot_mat <- matrix(nrow = length(beta), ncol = 0)
  gamma_boot_mat <- matrix(nrow = length(gamma), ncol = 0)
  alpha_boot_mat <- matrix(nrow = length(alpha), ncol = 0)
  se_boot_mat <- matrix(nrow = length(se_plot), ncol = 0)
  AUC_boot <- c()

  for (iter in 1: n_boot){
    stop <- FALSE
    # Prevent errors from being reported due to too many duplicate samples
    while (!stop){
      index <- sample(rownames(data), nrow(data), replace = TRUE)
      index_d <- sample(rownames(data_d), nrow(data_d), replace = TRUE)

      result <- tryCatch({
        par_list_boot <- odpROC_(vars, eta_vars, data[index, ], data_d[index_d, ],
                                 response, link, base, Intercept, eta, link_ordinal)
        1
      }, warning = function(w){
        2
      }, error = function(e){
        3
      }, finally = {})

      if (result == 1){
        stop = TRUE
      }
    }

    beta_boot <- par_list_boot$beta
    gamma_boot <- par_list_boot$gamma
    alpha_boot <- par_list_boot$alpha

    beta_boot_mat <- cbind(beta_boot_mat, beta_boot)
    gamma_boot_mat <- cbind(gamma_boot_mat, gamma_boot)
    alpha_boot_mat <- cbind(alpha_boot_mat, alpha_boot)
    se_boot_mat <- cbind(se_boot_mat, cal_ROC(fpr_plot, beta_boot, gamma_boot, alpha_boot))
    AUC_boot <- c(AUC_boot, integrate(cal_ROC, 0.0, 1.0, beta_boot, gamma_boot, alpha_boot)$value)
  }

  # Print the estimates of the parameters along with the standard deviation
  par_info <- matrix(nrow = 0, ncol = 2)
  colnames(par_info) <- c("Estimate", "std. Error")
  if (length(alpha) != 0){
    alpha_std <- sqrt(apply(alpha_boot_mat, MARGIN = 1, var))
    par_info <- rbind(par_info, cbind(alpha, alpha_std))
  }
  if (length(beta) != 0){
    beta_std <- sqrt(apply(beta_boot_mat, MARGIN = 1, var))
    par_info <- rbind(par_info, cbind(beta, beta_std))
  }
  if (length(gamma) != 0){
    gamma_std <- sqrt(apply(gamma_boot_mat, MARGIN = 1, var))
    par_info <- rbind(par_info, cbind(gamma, gamma_std))
  }

  # Print the estimate of AUC along with the standard deviation
  AUC_std <- sqrt(var(AUC_boot))
  AUC_info <- cbind(AUC, AUC_std)
  colnames(AUC_info) <- c("Estimate", "std. Error")
  rownames(AUC_info) <- c("AUC")

  cat("Coefficients: \n")
  print(par_info, digits = 4)
  cat("\n")
  cat("AUC: \n")
  print(AUC_info, digits = 4)
  cat("\n")

  # plot
  se_std <- sqrt(apply(se_boot_mat, MARGIN = 1, var))
  plot(se_plot ~ fpr_plot, type = "l", main = "ROC Curve Using Parametric Direct Regression Model",
       xlim = c(0.0, 1.0), ylim = c(0.0, 1.0), xlab = "FPR", ylab = "TPR")
  lines(se_plot - qnorm((1 + level) / 2) * se_std ~ fpr_plot, lty = 2)
  lines(se_plot + qnorm((1 + level) / 2) * se_std ~ fpr_plot, lty = 2)
  legend("bottomright", legend = c("ROC Curve", "CI"), lty = c(1, 2))

  list_return <- list(beta = beta, gamma = gamma, alpha = alpha, AUC = AUC)
  return(list_return)
}


# Auxiliary function, used for parameter and ROC curve estimation
odpROC_ <- function(vars, eta_vars, data, data_d, response, link, base,
                    Intercept, eta, link_ordinal){

  # Calculate the distribution function and density function corresponding to "link"
  func <- function(x, type){
    if (link == "logit"){
      expx = exp(x)
    }
    if (type == "p"){
      return(switch(link, "logit" = expx / (1 + expx),
                    "probit" = pnorm(x),
                    "cauchit" = 0.5 + 1 / pi * atan(x)))
    }
    if (type == "d"){
      return(switch(link, "logit" = expx / (1 + expx)^2,
                    "probit" = dnorm(x),
                    "cauchit" = 1 / (pi * (1 + x^2))))
    }
    if (type == "dd"){
      return(switch(link, "logit" = expx * (1 - expx) / (1 + expx)^3,
                    "probit" = -x * dnorm(x),
                    "cauchit" = -2 * x / (pi * (1 + x^2)^2)))
    }
  }

  # Generate the matrices needed to estimate the parameters
  mat <- as.matrix(data_d[vars])
  mat_eta <- as.matrix(data_d[eta_vars])
  n <- nrow(data)
  n_d <- nrow(data_d)

  a <- length(vars)
  b <- length(eta_vars)
  c <- length(base) + Intercept

  # Formula for ordinal regression
  terms = intersect(colnames(data), vars)
  if (length(terms) == 0){
    formula <- as.formula(paste(response, "~1"))
  }
  else {
    formula <- reformulate(terms, response = response)
  }

  # U_ij = 1 if and only if T_i >= T_j
  U_ij <- data_d[[response]] >= tcrossprod(rep(1, n_d), data[[response]])

  # Use ordinal regression to get an estimate of the empirical survival function and thresholds
  # Use this estimate to obtain p_ij
  data[[response]] <- factor(data[[response]])
  clm_model <- clm(formula = formula, scale = ~1, data = data, link = link_ordinal)
  cum_prob <- predict(clm_model, data_d[terms], type = "cum.prob")[[2]]
  p_ij <- 1 - cum_prob[1: n_d, data[[response]]]

  # Mark as valid the values that satisfy 0 < p_ij < 1
  invalid <- (p_ij == 0) | (p_ij == 1)
  p_ij[invalid] <- 0.5

  # Generate a list of matrices and matrix corresponding to base and eta
  p_mat_eta <- make.link(eta)$linkfun(p_ij)
  list_mat_base <- list()
  if (Intercept){
    list_mat_base <- append(list_mat_base, list(matrix(1, nrow = n_d, ncol = n)))
  }
  for (bs in base){
    list_mat_base <- append(list_mat_base, list(make.link(bs)$linkfun(p_ij)))
  }

  # score function
  score <- function(x){
    y <- numeric(a + b + c)
    W <- numeric(n_d)

    if (a != 0){
      W <- W + as.vector(mat %*% x[1: a])
    }
    if (b != 0){
      W <- W + as.vector(mat_eta %*% x[(a + 1): (a + b)]) * p_mat_eta
    }
    if (c != 0){
      for (i in 1: c){
        W <- W + x[(a + b + i)] * list_mat_base[[i]]
      }
    }

    tmp <- func(W, "d") / (func(W, "p") - 1 + U_ij)
    tmp[invalid] <- 0

    if (a != 0){
      y[1: a] <- crossprod(mat, rowSums(tmp))
    }
    if (b != 0){
      y[(a + 1): (a + b)] <- rowSums(crossprod(mat_eta, tmp * p_mat_eta))
    }
    if (c != 0){
      for (i in 1: c){
        y[(a + b + i)] <- sum((tmp * list_mat_base[[i]]))
      }
    }

    return(y)
  }

  # Solve the system of equations using nleqslv to obtain estimates of the parameters.
  x0 <- numeric(a + b + c)
  result <- nleqslv(x = x0, fn = score)
  par <- result$x

  list_return <- list(beta = c(), gamma = c(), alpha = c())
  if (a != 0){
    list_return$beta <- par[1: a]
  }
  if (b != 0){
    list_return$gamma <- par[(a + 1): (a + b)]
  }
  if (c != 0){
    list_return$alpha <- par[(a + b + 1): (a + b + c)]
  }

  return(list_return)
}
