#' @title Indirect Regression Model for Continuous-Scale ROC Curves
#'
#' @description you can use this function to fit an indirect regression model for
#'     continuous-scale ROC curves. This function will print and return the parameters
#'     of the model and the AUC of the corresponding ROC curve, and will make an
#'     image of the ROC curve and its confidence band.
#'
#' @param location a two-sided formula with the response variable on the LHS and
#'     the predictor for the mean on the RHS.
#' @param scale a one-sided formula with the predictor for the standard deviation on the RHS.
#' @param data a data frame in which to evaluate the indirect regression model.
#' @param newdata a data frame for generating ROC curve.
#' @param group a character representing the name of the column in data that
#'     represents the true condition of subjects.
#' @param level the confidence level. Default 0.95.
#' @param n_boot the number of bootstrap samples to draw. Default 1000.
#'
#' @import lmls
#' @import stats
#' @import graphics
#'
#' @return a list of the parameters of the model and the AUC of the corresponding ROC curve.
#'
#' @examples
#' \donttest{
#' data("data_continous")
#' ciROC(location = t ~ d * (x1 + x2),
#'       scale = ~ d * (x1 + x2),
#'       data = data_continous,
#'       newdata = data.frame(x1 = c(0), x2 = c(0)),
#'       group="d")
#' }
#'
#' @export

ciROC <- function(location, scale, data, newdata, group,
                  level = 0.95, n_boot = 1000){

  # Generate new data for calculating ROC curve
  newdata_0 <- newdata
  newdata_0[[group]] <- 0
  newdata_1 <- newdata
  newdata_1[[group]] <- 1

  # empirical survival function
  survival <- function(t, res){
    return(rowSums(tcrossprod(rep(1, length(t)), res) - t >= 0) / length(res))
  }

  # inverse empirical survival function
  inv_survival <- function(p, res){
    res <- sort(res, decreasing = TRUE)
    y <- numeric(length(p))
    y[p == 0] = Inf
    y[p > 0] = res[ceiling(p * length(res))]
    return(y)
  }

  # Calculate the value of the ROC curve
  cal_ROC <- function(p, model){
    a <- (predict(model, newdata_1, predictor = "location") -
            predict(model, newdata_0, predictor = "location")) /
      predict(model, newdata_1, predictor = "scale", type = "response")
    b <- predict(model, newdata_0, predictor = "scale", type = "response") /
      predict(model, newdata_1, predictor = "scale", type = "response")

    res <- residuals(model)
    se <- survival(b * inv_survival(p, res) - a, res)
    return(se)
  }

  # Calculate AUC
  cal_AUC <- function(model){
    a <- (predict(model, newdata_1, predictor = "location") -
            predict(model, newdata_0, predictor = "location")) /
      predict(model, newdata_1, predictor = "scale", type = "response")
    b <- predict(model, newdata_0, predictor = "scale", type = "response") /
      predict(model, newdata_1, predictor = "scale", type = "response")

    res <- residuals(model)
    p <- sort(unique(survival(res, res)))
    se <- survival(b * inv_survival(p, res) - a, res)
    dp <- p - c(0, p[1: length(p) - 1])
    return(crossprod(se, dp))
  }

  # Fit a semi-parametric location-scale regression
  model <- lmls(location, scale, data)
  model[["call"]][["location"]] <- location
  model[["call"]][["scale"]] <- scale

  loc_coef <- model[["coefficients"]][["location"]]
  scale_coef <- model[["coefficients"]][["scale"]]
  res <- residuals(model)

  # Generate fpr_plot for mapping, calculate se_plot and AUC
  fpr_plot <- seq(0, 1, 0.001)
  se_plot <- cal_ROC(fpr_plot, model)
  AUC <- cal_AUC(model)

  # Use bootstrap to calculate parameters, standard deviation of ROC curves and AUCs
  loc_boot_mat <- matrix(nrow = length(loc_coef), ncol = 0)
  scale_boot_mat <- matrix(nrow = length(scale_coef), ncol = 0)
  se_boot_mat <- matrix(nrow = length(se_plot), ncol = 0)
  AUC_boot <- c()

  for (iter in 1: n_boot){
    stop <- FALSE
    # Prevent errors from being reported due to too many duplicate samples
    while (!stop){
      index <- sample(rownames(data), nrow(data), replace = TRUE)

      result <- tryCatch({
        model_boot <- lmls(location, scale, data[index, ])
        1
      }, warning = function(w){
        2
      }, error = function(e){
        3
      }, finally = {})

      if (result == 1){
        stop = TRUE
      }
    }

    model_boot[["call"]][["location"]] <- location
    model_boot[["call"]][["scale"]] <- scale

    loc_boot_mat <- cbind(loc_boot_mat, model_boot[["coefficients"]][["location"]])
    scale_boot_mat <- cbind(scale_boot_mat, model_boot[["coefficients"]][["scale"]])
    se_boot_mat <- cbind(se_boot_mat, cal_ROC(fpr_plot, model_boot))
    AUC_boot <- c(AUC_boot, cal_AUC(model_boot))
  }

  # Print the estimates of the parameters along with the standard deviation
  loc_std <- sqrt(apply(loc_boot_mat, MARGIN = 1, var))
  scale_std <- sqrt(apply(scale_boot_mat, MARGIN = 1, var))
  se_std <- sqrt(apply(se_boot_mat, MARGIN = 1, var))
  AUC_std <- sqrt(var(AUC_boot))

  loc_info <- cbind(loc_coef, loc_std)
  scale_info <- cbind(scale_coef, scale_std)
  AUC_info <- cbind(AUC, AUC_std)

  colnames(loc_info) <- c("Estimate", "std. Error")
  colnames(scale_info) <- c("Estimate", "std. Error")
  colnames(AUC_info) <- c("Estimate", "std. Error")
  rownames(AUC_info) <- c("AUC")

  cat("Location coefficients: \n")
  print(loc_info, digits = 4)
  cat("\n")
  cat("Scale coefficients (log link): \n")
  print(scale_info, digits = 4)
  cat("\n")
  cat("AUC: \n")
  print(AUC_info, digits = 4)
  cat("\n")

  # plot
  plot(se_plot ~ fpr_plot, type = "l", main = "ROC Curve Using Indirect Regression Model",
       xlim = c(0.0, 1.0), ylim = c(0.0, 1.0), xlab = "FPR", ylab = "TPR")
  lines(se_plot - qnorm((1 + level) / 2) * se_std ~ fpr_plot, lty = 2)
  lines(se_plot + qnorm((1 + level) / 2) * se_std ~ fpr_plot, lty = 2)
  legend("bottomright", legend = c("ROC Curve", "CI"), lty = c(1, 2))

  list_return <- list(location = loc_coef, scale = scale_coef, AUC = AUC, res = res)
  return(list_return)
}
