#' @title Direct Parametric Regression Model for Continuous-Scale ROC Curves
#'
#' @description you can use this function to fit an direct parametric regression
#'     model for continuous-scale ROC curves. This function will print and return
#'     the parameters of the model and the AUC of the corresponding ROC curve,
#'     and will make an image of the ROC curve and its confidence band.
#'
#' @param vars a vector of characters representing the covariates in the model.
#' @param eta_vars a vector of characters representing the covariates interacting
#'     with functions of p in the model.
#' @param data a data frame consisting of data from subjects without condition.
#' @param data_d a data frame consisting of data from subjects with condition.
#' @param newdata_d a data frame for generating ROC curve.
#' @param response a character representing the name of the column in both data
#'     and data_d that represents the test results of subjects.
#' @param link link function in direct regression model. "probit", "logit" or "cauchit".
#'     Default "probit".
#' @param base base functions in direct regression model. A vector consisting of
#'     one or more of "probit", "logit" and "cauchit".
#' @param Intercept indicates whether the model contains a constant term. Default TRUE.
#' @param eta function interacting with eta_vars. "probit", "logit" or "cauchit".
#'     Default "probit".
#' @param p_lower the lower bound of fprs that users are interested in. Default 0.0.
#' @param p_upper the upper bound of fprs that users are interested in. Default 1.0.
#' @param n_p the number of fprs for estimating the regression model. Default 50.
#' @param level the confidence level. Default 0.95.
#' @param n_boot the number of bootstrap samples to draw. Default 1000.
#'
#' @import quantreg
#' @import nleqslv
#' @import stats
#' @import graphics
#'
#' @return a list of the parameters of the model and the AUC of the corresponding ROC curve.
#'
#' @examples
#' \donttest{
#' data("data_continous")
#' data <- data_continous[data_continous[["d"]] == 0, c("t", "x1", "x2")]
#' data_d <- data_continous[data_continous[["d"]] == 1, c("t", "x1", "x2")]
#' cdpROC(vars = c("x1", "x2"),
#'        eta_vars = c(),
#'        data = data,
#'        data_d = data_d,
#'        newdata_d = data.frame(x1 = c(0), x2 = c(0)),
#'        response = "t")
#' }
#'
#' @export

cdpROC <- function(vars, eta_vars, data, data_d, newdata_d, response,
                   link = "probit", base = c("probit"), Intercept = TRUE, eta = "probit",
                   p_lower = 0.0, p_upper = 1.0, n_p = 50, level = 0.95, n_boot = 1000){

  # Generate fpr_set required to perform parameter estimation
  fpr_set <- seq(1, n_p) / (1 + n_p)
  fpr_set <- fpr_set[(fpr_set >= p_lower) & (fpr_set <= p_upper)]
  base <- unique(base)

  # Calculate the value of the ROC curve
  cal_ROC <- function(p, beta, gamma, alpha){
    W_new <- sum(beta * newdata_d[vars])
    W_eta_new <- sum(gamma * newdata_d[eta_vars])

    vec_eta_new <- make.link(eta)$linkfun(p)
    mat_base_new <- matrix(nrow = length(p), ncol = 0)
    if (Intercept){
      mat_base_new <- cbind(mat_base_new, rep(1, length(p)))
    }
    for (bs in base){
      mat_base_new <- cbind(mat_base_new, make.link(bs)$linkfun(p))
    }

    return(make.link(link)$linkinv(W_new + W_eta_new * vec_eta_new +
                        as.vector(mat_base_new %*% alpha)))
  }

  # Use auxiliary function cdpROC_ to get parameter estimates
  par_list <- cdpROC_(vars, eta_vars, data, data_d, response, link, base,
                      Intercept, eta, fpr_set)
  beta <- par_list$beta
  gamma <- par_list$gamma
  alpha <- par_list$alpha

  if (length(beta) != 0){
    names(beta) <- vars
  }
  if (length(gamma) != 0){
    names(gamma) <- paste(eta_vars, eta, sep = ":")
  }

  if (Intercept){
    names(alpha) <- c("(Intercept)", base)
  }
  else {
    if (length(base) != 0){
      names(alpha) <- base
    }
  }

  # Generate fpr_plot for mapping, calculate se_plot and AUC
  fpr_plot <- seq(0.001, 0.999, 0.001)
  fpr_plot <- fpr_plot[(fpr_plot >= p_lower) & (fpr_plot <= p_upper)]
  se_plot <- cal_ROC(fpr_plot, beta, gamma, alpha)
  AUC <- integrate(cal_ROC, p_lower, p_upper, beta, gamma, alpha)$value

  # Use bootstrap to calculate parameters, standard deviation of ROC curves and AUCs
  beta_boot_mat <- matrix(nrow = length(beta), ncol = 0)
  gamma_boot_mat <- matrix(nrow = length(gamma), ncol = 0)
  alpha_boot_mat <- matrix(nrow = length(alpha), ncol = 0)
  se_boot_mat <- matrix(nrow = length(se_plot), ncol = 0)
  AUC_boot <- c()

  for (iter in 1: n_boot){
    stop <- FALSE
    # Prevent errors from being reported due to too many duplicate samples
    while (!stop){
      index <- sample(rownames(data), nrow(data), replace = TRUE)
      index_d <- sample(rownames(data_d), nrow(data_d), replace = TRUE)

      result <- tryCatch({
        par_list_boot <- cdpROC_(vars, eta_vars, data[index, ], data_d[index_d, ],
                                 response, link, base, Intercept, eta, fpr_set)
        1
      }, warning = function(w){
        2
      }, error = function(e){
        3
      }, finally = {})

      if (result == 1){
        stop = TRUE
      }
    }

    beta_boot <- par_list_boot$beta
    gamma_boot <- par_list_boot$gamma
    alpha_boot <- par_list_boot$alpha

    beta_boot_mat <- cbind(beta_boot_mat, beta_boot)
    gamma_boot_mat <- cbind(gamma_boot_mat, gamma_boot)
    alpha_boot_mat <- cbind(alpha_boot_mat, alpha_boot)
    se_boot_mat <- cbind(se_boot_mat, cal_ROC(fpr_plot, beta_boot, gamma_boot, alpha_boot))
    AUC_boot <- c(AUC_boot, integrate(cal_ROC, p_lower, p_upper, beta_boot, gamma_boot, alpha_boot)$value)
  }

  # Print the estimates of the parameters along with the standard deviation
  par_info <- matrix(nrow = 0, ncol = 2)
  colnames(par_info) <- c("Estimate", "std. Error")
  if (length(alpha) != 0){
    alpha_std <- sqrt(apply(alpha_boot_mat, MARGIN = 1, var))
    par_info <- rbind(par_info, cbind(alpha, alpha_std))
  }
  if (length(beta) != 0){
    beta_std <- sqrt(apply(beta_boot_mat, MARGIN = 1, var))
    par_info <- rbind(par_info, cbind(beta, beta_std))
  }
  if (length(gamma) != 0){
    gamma_std <- sqrt(apply(gamma_boot_mat, MARGIN = 1, var))
    par_info <- rbind(par_info, cbind(gamma, gamma_std))
  }

  # Print the estimate of AUC along with the standard deviation
  AUC_std <- sqrt(var(AUC_boot))
  AUC_info <- cbind(AUC, AUC_std)
  colnames(AUC_info) <- c("Estimate", "std. Error")
  rownames(AUC_info) <- c("AUC")

  cat("Coefficients: \n")
  print(par_info, digits = 4)
  cat("\n")
  cat("AUC: \n")
  print(AUC_info, digits = 4)
  cat("\n")

  # plot
  se_std <- sqrt(apply(se_boot_mat, MARGIN = 1, var))
  plot(se_plot ~ fpr_plot, type = "l", main = "ROC Curve Using Parametric Direct Regression Model",
       xlim = c(p_lower, p_upper), ylim = c(0.0, 1.0), xlab = "FPR", ylab = "TPR")
  lines(se_plot - qnorm((1 + level) / 2) * se_std ~ fpr_plot, lty = 2)
  lines(se_plot + qnorm((1 + level) / 2) * se_std ~ fpr_plot, lty = 2)
  legend("bottomright", legend = c("ROC Curve", "CI"), lty = c(1, 2))

  list_return <- list(beta = beta, gamma = gamma, alpha = alpha, AUC = AUC)
  return(list_return)
}


# Auxiliary function, used for parameter and ROC curve estimation
cdpROC_ <- function(vars, eta_vars, data, data_d, response, link, base,
                    Intercept, eta, fpr_set){

  # Calculate the distribution function and density function corresponding to "link"
  func <- function(x, type){
    if (link == "logit"){
      expx = exp(x)
    }
    if (type == "p"){
      return(switch(link, "logit" = expx / (1 + expx),
                    "probit" = pnorm(x),
                    "cauchit" = 0.5 + 1 / pi * atan(x)))
    }
    if (type == "d"){
      return(switch(link, "logit" = expx / (1 + expx)^2,
                    "probit" = dnorm(x),
                    "cauchit" = 1 / (pi * (1 + x^2))))
    }
    if (type == "dd"){
      return(switch(link, "logit" = expx * (1 - expx) / (1 + expx)^3,
                    "probit" = -x * dnorm(x),
                    "cauchit" = -2 * x / (pi * (1 + x^2)^2)))
    }
  }

  # Generate the matrices needed to estimate the parameters
  mat <- as.matrix(data_d[vars])
  mat_eta <- as.matrix(data_d[eta_vars])
  n_d <- nrow(data_d)

  a <- length(vars)
  b <- length(eta_vars)
  c <- length(base) + Intercept
  n_p <- length(fpr_set)

  # Generate matrices and vectors corresponding to base and eta
  vec_eta <- make.link(eta)$linkfun(fpr_set)
  mat_base <- matrix(nrow = n_p, ncol = 0)
  if (Intercept){
    mat_base <- cbind(mat_base, rep(1, n_p))
  }
  for (bs in base){
    mat_base <- cbind(mat_base, make.link(bs)$linkfun(fpr_set))
  }

  # Formula for quantile regression
  terms = intersect(colnames(data), vars)
  if (length(terms) == 0){
    rq_formula <- as.formula(paste(response, "1", sep = " ~ "))
  }
  else {
    rq_formula <- reformulate(terms, response = response)
  }

  # Use quantile regression to obtain placement values and indicator variable U_ip
  # U_ip is an n_d * n_p matrix, where U_ip = 1 if and only if T_i <= p
  U_ip = matrix(nrow = n_d, ncol = 0)
  for (fpr in fpr_set){
    model <- rq(rq_formula, 1 - 1e-6 - fpr, data)
    U_ip <- cbind(U_ip, as.numeric(data_d[[response]] >= predict(model, data_d)))
  }

  # score function
  score <- function(x){
    y <- numeric(a + b + c)
    W <- numeric(n_d)

    if (a != 0){
      W <- W + as.vector(mat %*% x[1: a])
    }
    if (b != 0){
      W <- W + tcrossprod(mat_eta %*% x[(a + 1): (a + b)], vec_eta)
    }
    if (c != 0){
      W <- W + tcrossprod(rep(1, n_d), mat_base %*% x[(a + b + 1): (a + b + c)])
    }

    tmp <- func(W, "d") / (func(W, "p") - 1 + U_ip)
    if (a != 0){
      y[1: a] <- crossprod(mat, rowSums(tmp))
    }
    if (b != 0){
      y[(a + 1): (a + b)] <- crossprod(mat_eta, tmp) %*% vec_eta
    }
    if (c != 0){
      y[(a + b + 1): (a + b + c)] <- crossprod(mat_base, colSums(tmp))
    }

    return(y)
  }

  # Jacobian matrix
  jac <- function(x){
    I <-matrix(nrow = a + b + c, ncol = a + b + c)
    W <- numeric(n_d)

    if (a != 0){
      W <- W + as.vector(mat %*% x[1: a])
    }
    if (b != 0){
      W <- W + tcrossprod(mat_eta %*% x[(a + 1): (a + b)], vec_eta)
    }
    if (c != 0){
      W <- W + tcrossprod(rep(1, n_d), mat_base %*% x[(a + b + 1): (a + b + c)])
    }

    tmp <- func(W, "p") - 1 + U_ip
    tmp <- func(W, "dd") / tmp - (func(W, "d") / tmp)^2

    if (a != 0){
      I[1: a, 1: a] <- crossprod(mat, diag(rowSums(tmp)) %*% mat)
      if (b != 0){
        I[1: a, (a + 1): (a + b)] <-
          crossprod(mat, diag(as.vector(tmp %*% vec_eta)) %*% mat_eta)
        I[(a + 1): (a + b), 1: a] <- t(I[1: a, (a + 1): (a + b)])
      }
      if (c != 0){
        I[1: a, (a + b + 1): (a + b + c)] <-
          crossprod(mat, tmp %*% mat_base)
        I[(a + b + 1): (a + b + c), 1: a] <- t(I[1: a, (a + b + 1): (a + b + c)])
      }
    }
    if (b != 0){
      I[(a + 1): (a + b), (a + 1): (a + b)] <-
        crossprod(mat_eta, diag(as.vector(tmp %*% (vec_eta^2))) %*% mat_eta)
      if (c != 0){
        I[(a + 1): (a + b), (a + b + 1): (a + b + c)] <-
          crossprod(mat_eta, tmp %*% diag(vec_eta) %*% mat_base)
        I[(a + b + 1): (a + b + c), (a + 1): (a + b)] <-
          t(I[(a + 1): (a + b), (a + b + 1): (a + b + c)])
      }
    }
    if (c != 0){
      I[(a + b + 1): (a + b + c), (a + b + 1): (a + b + c)] <-
        crossprod(mat_base, diag(colSums(tmp)) %*% mat_base)
    }

    return(I)
  }

  # Solve the system of equations using nleqslv to obtain estimates of the parameters.
  x0 <- numeric(a + b + c)
  result <- nleqslv(x = x0, fn = score, jac = jac, method = "Newton")
  par <- result$x

  list_return <- list(beta = c(), gamma = c(), alpha = c())
  if (a != 0){
    list_return$beta <- par[1: a]
  }
  if (b != 0){
    list_return$gamma <- par[(a + 1): (a + b)]
  }
  if (c != 0){
    list_return$alpha <- par[(a + b + 1): (a + b + c)]
  }

  return(list_return)
}

