library(shiny)
library(readr)
library(readxl)
library(ggplot2)

# 加权标准差计算的 Server
wsd_server <- function(input, output, session) {
  
  observeEvent(input$run_wsd, {
    req(input$wsd_csv_input || input$wsd_excel_input)
    
    # 读取数据
    data <- NULL
    if (!is.null(input$wsd_csv_input)) {
      data <- read_csv(input$wsd_csv_input$datapath)
    } else if (!is.null(input$wsd_excel_input)) {
      data <- read_excel(input$wsd_excel_input$datapath)
    }
    
    # 预处理数据
    processed_data <- as.matrix(data)  # 确保数据是矩阵格式
    
    # 计算每个患者的加权标准差 (wSDi) 和整体的 wSD, lwSD, rwSD
    wsd_results <- sapply(1:nrow(processed_data), function(i) wSDi(processed_data, i))
    overall_wsd <- wSD(processed_data)
    
    # 输出结果
    output$wsd_results <- renderPrint({
      cat("每个患者的加权标准差 (wSDi):\n")
      print(wsd_results)
      cat("\n整体加权标准差 (wSD):", overall_wsd[1], "\n")
      cat("左界限 (lwSD):", overall_wsd[2], "\n")
      cat("右界限 (rwSD):", overall_wsd[3], "\n")
    })
    
    # 绘图输出 - 频率直方图
    output$wsd_plot <- renderPlot({
      ggplot(data.frame(WSDi = wsd_results), aes(x = WSDi)) +
        geom_histogram(binwidth = 0.1, fill = "blue", color = "white", alpha = 0.7) +
        labs(title = "加权标准差 (wSDi) 频率直方图", x = "加权标准差 (wSDi)", y = "频率") +
        theme_minimal()
    })
  })
  
  observeEvent(input$default_wsd, {
    # 使用默认数据
    n_patients <- 20  # 增加患者数量
    n_measurements <- 5  # 增加测量次数（列数）
    
    # 生成示例数据，确保每个患者的数据不同
    set.seed(123)  # 为可重复性设置随机种子
    data <- matrix(rnorm(n_patients * n_measurements, mean = 0, sd = 1), nrow = n_patients, byrow = TRUE)
    
    # 确保数据是矩阵格式
    data <- as.matrix(data)
    
    # 计算每个患者的加权标准差 (wSDi) 和整体的 wSD, lwSD, rwSD
    wsd_results <- sapply(1:n_patients, function(i) wSDi(data, i))
    overall_wsd <- wSD(data)
    
    # 输出结果
    output$wsd_results <- renderPrint({
      cat("每个患者的加权标准差 (wSDi):\n")
      print(wsd_results)
      cat("\n整体加权标准差 (wSD):", overall_wsd[1], "\n")
      cat("左界限 (lwSD):", overall_wsd[2], "\n")
      cat("右界限 (rwSD):", overall_wsd[3], "\n")
    })
    
    # 绘图输出 - 频率直方图
    output$wsd_plot <- renderPlot({
      ggplot(data.frame(WSDi = wsd_results), aes(x = WSDi)) +
        geom_histogram(binwidth = 0.1, fill = "blue", color = "white", alpha = 0.7) +
        labs(title = "加权标准差 (wSDi) 频率直方图", x = "加权标准差 (wSDi)", y = "频率") +
        theme_minimal()
    })
    
    # 显示默认数据
    output$default_wsd_data_display <- renderPrint({
      cat("默认数据示例（矩阵形式）：\n")
      print(data)  # 以矩阵形式展示数据
    })
  })
}








# 第二个子页面的 Server 逻辑
rc_server <- function(input, output, session) {
  
  observeEvent(input$run_rc, {
    req(input$rc_csv_input || input$rc_excel_input)
    
    # 读取数据
    data <- NULL
    if (!is.null(input$rc_csv_input)) {
      data <- read_csv(input$rc_csv_input$datapath)
    } else if (!is.null(input$rc_excel_input)) {
      data <- read_excel(input$rc_excel_input$datapath)
    }
    
    # 预处理数据
    processed_data <- as.matrix(data)
    
    # 计算重复性系数
    rc_results <- RC(processed_data)
    
    # 输出结果
    output$rc_results <- renderPrint({
      cat("重复性系数 (RC):", rc_results[1], "\n")
      cat("左置信区间 (lRC):", rc_results[2], "\n")
      cat("右置信区间 (rRC):", rc_results[3], "\n")
    })
  })
  
  observeEvent(input$default_rc, {
    # 使用默认数据
    n_patients <- 20
    n_measurements <- 5
    
    set.seed(123)
    data <- matrix(rnorm(n_patients * n_measurements, mean = 0, sd = 1), nrow = n_patients, byrow = TRUE)
    
    # 计算重复性系数
    processed_data <- as.matrix(data)
    rc_results <- RC(processed_data)
    
    # 输出结果
    output$rc_results <- renderPrint({
      cat("使用默认数据计算的重复性系数 (RC):", rc_results[1], "\n")
      cat("左置信区间 (lRC):", rc_results[2], "\n")
      cat("右置信区间 (rRC):", rc_results[3], "\n")
    })
    
    # 显示默认数据结构
    output$default_rc_data_display <- renderPrint({
      cat("默认数据示例（矩阵形式）：\n")
      print(data)
    })
  })
}




# 第三个子页面的 Server 逻辑
wcv_server <- function(input, output, session) {
  
  observeEvent(input$run_wcv, {
    req(input$wcv_csv_input || input$wcv_excel_input)
    
    # 读取数据
    data <- NULL
    if (!is.null(input$wcv_csv_input)) {
      data <- read_csv(input$wcv_csv_input$datapath)
    } else if (!is.null(input$wcv_excel_input)) {
      data <- read_excel(input$wcv_excel_input$datapath)
    }
    
    # 预处理数据
    processed_data <- as.matrix(data)
    
    # 计算被试内变异系数
    wcv_results <- wCV(processed_data)
    wCVc_result <- wCVc(processed_data, input$beta1, input$beta0)  # 计算 wCVc
    
    # 输出结果
    output$wcv_results <- renderPrint({
      cat("被试内变异系数 (wCV):", wcv_results[1], "\n")
      cat("左置信区间 (lwCV):", wcv_results[2], "\n")
      cat("右置信区间 (rwCV):", wcv_results[3], "\n")
      cat("加权变异系数 (wCVc):", wCVc_result, "\n")
    })
  })
  
  observeEvent(input$default_wcv, {
    # 使用默认数据
    n_patients <- 20
    n_measurements <- 5
    
    set.seed(123)
    data <- matrix(rnorm(n_patients * n_measurements, mean = 0, sd = 1), nrow = n_patients, byrow = TRUE)
    
    # 计算被试内变异系数和加权变异系数
    processed_data <- as.matrix(data)
    wcv_results <- wCV(processed_data)
    wCVc_result <- wCVc(processed_data, beta1 = 1, beta0 = 0)  # 默认参数值
    
    # 输出结果
    output$wcv_results <- renderPrint({
      cat("使用默认数据计算的被试内变异系数 (wCV):", wcv_results[1], "\n")
      cat("左置信区间 (lwCV):", wcv_results[2], "\n")
      cat("右置信区间 (rwCV):", wcv_results[3], "\n")
      cat("加权变异系数 (wCVc):", wCVc_result, "\n")
    })
    
    # 显示默认数据结构
    output$default_wcv_data_display <- renderPrint({
      cat("默认数据示例（矩阵形式,beta1 = 1, beta0 = 0）：\n")
      print(data)
    })
  })
}











# 第四个子页面的 Server 逻辑
bias_server <- function(input, output, session) {
  
  observeEvent(input$run_bias, {
    req(input$bias_csv_input || input$bias_excel_input)
    
    # 读取数据
    data <- NULL
    if (!is.null(input$bias_csv_input)) {
      data <- read_csv(input$bias_csv_input$datapath)
    } else if (!is.null(input$bias_excel_input)) {
      data <- read_excel(input$bias_excel_input$datapath)
    }
    
    # 预处理数据
    processed_data <- as.matrix(data)
    
    # 处理真实期望 X
    X <- as.numeric(unlist(strsplit(input$X_values, ",")))
    
    # 计算常数偏差
    bias_results <- D(processed_data, X)
    varD_results <- VarD(processed_data, X)
    
    # 输出结果
    output$bias_results <- renderPrint({
      cat("常数偏差 D:", bias_results[1], "\n")
      cat("偏差的百分比:", bias_results[2], "\n")
      cat("偏差方差 VarD:", varD_results[1], "\n")
      cat("左置信区间 (lD):", varD_results[2], "\n")
      cat("右置信区间 (rD):", varD_results[3], "\n")
    })
  })
  
  observeEvent(input$default_bias, {
    # 使用默认数据
    n_patients <- 5
    n_measurements <- 3
    true_expectations <- c(10, 10, 10, 10, 10)  # 默认的真实期望值
    
    set.seed(123)
    data <- matrix(rnorm(n_patients * n_measurements, mean = 10, sd = 2), nrow = n_patients, byrow = TRUE)
    
    # 计算常数偏差
    processed_data <- as.matrix(data)
    bias_results <- D(processed_data, true_expectations)
    varD_results <- VarD(processed_data, true_expectations)
    
    # 输出结果
    output$bias_results <- renderPrint({
      cat("使用默认数据计算的常数偏差 D:", bias_results[1], "\n")
      cat("偏差的百分比:", bias_results[2], "\n")
      cat("偏差方差 VarD:", varD_results[1], "\n")
      cat("左置信区间 (lD):", varD_results[2], "\n")
      cat("右置信区间 (rD):", varD_results[3], "\n")
    })
    
    # 显示默认数据结构
    output$default_bias_data_display <- renderPrint({
      cat("默认数据示例（矩阵形式，真实被测量值为（10, 10, 10, 10, 10））：\n")
      print(data)
    })
  })
}












# 第五个子页面的 Server 逻辑
ybias_server <- function(input, output, session) {
  
  observeEvent(input$run_ybias, {
    req(input$ybias_csv_input || input$ybias_excel_input)
    
    # 读取数据
    data <- NULL
    if (!is.null(input$ybias_csv_input)) {
      data <- read_csv(input$ybias_csv_input$datapath)
    } else if (!is.null(input$ybias_excel_input)) {
      data <- read_excel(input$ybias_excel_input$datapath)
    }
    
    # 预处理数据
    processed_data <- as.matrix(data)
    
    # 处理真实期望 X
    X <- as.numeric(unlist(strsplit(input$X_values_ybias, ",")))
    
    # 计算非恒定偏差
    ybias_results <- Ybias(processed_data, X)
    
    # 输出结果
    output$ybias_results <- renderPrint({
      cat("左置信区间 (lY):\n")
      print(ybias_results$lY)
      cat("右置信区间 (rY):\n")
      print(ybias_results$rY)
    })
  })
  
  observeEvent(input$default_ybias, {
    # 使用默认数据
    n_patients <- 5
    n_measurements <- 3
    
    # 为每个患者生成不同的真实均值
    true_expectations <- c(10, 12, 11, 9, 13)  # 每个患者的真实期望值
    set.seed(123)
    
    # 根据真实均值生成每个患者的测量数据
    data <- sapply(true_expectations, function(mean) rnorm(n_measurements, mean = mean, sd = 2))
    data <- t(data)
    
    # 计算非恒定偏差
    processed_data <- as.matrix(data)
    ybias_results <- Ybias(processed_data, true_expectations)
    
    # 输出结果
    output$ybias_results <- renderPrint({
      cat("使用默认数据计算的非恒定偏差结果：\n")
      cat("左置信区间 (lY):\n")
      print(ybias_results$lY)
      cat("右置信区间 (rY):\n")
      print(ybias_results$rY)
    })
    
    # 显示默认数据结构
    output$default_ybias_data_display <- renderPrint({
      cat("默认数据示例（矩阵形式，真实被测量值为（10, 12, 11, 9, 13））：\n")
      print(processed_data)
    })
  })
}








# 第六个子页面的 Server 逻辑
cot_server <- function(input, output, session) {
  
  observeEvent(input$run_cot, {
    req(input$cot_csv_input || input$cot_excel_input)
    
    # 读取数据
    data <- NULL
    if (!is.null(input$cot_csv_input)) {
      data <- read_csv(input$cot_csv_input$datapath)
    } else if (!is.null(input$cot_excel_input)) {
      data <- read_excel(input$cot_excel_input$datapath)
    }
    
    # 预处理数据
    processed_data <- as.matrix(data)
    
    # 处理真实期望 X
    X <- as.numeric(unlist(strsplit(input$X_values_cot, ",")))
    
    # 计算时间变化估计
    cot_results <- CoT(processed_data, X)
    
    # 输出结果
    output$cot_results <- renderPrint({
      cat("时间变化估计 CoT:\n")
      print(cot_results$CoT)
      cat("左置信区间 (lCoT):\n")
      print(cot_results$lCoT)
      cat("右置信区间 (rCoT):\n")
      print(cot_results$rCoT)
    })
  })
  
  observeEvent(input$default_cot, {
    # 使用默认数据
    n_patients <- 5
    n_measurements <- 3
    
    # 为每个患者生成不同的真实均值
    true_expectations <- c(10, 12, 11, 9, 13)  # 每个患者的真实期望值
    set.seed(123)
    
    # 根据真实均值生成每个患者的测量数据
    data <- sapply(true_expectations, function(mean) rnorm(n_measurements, mean = mean, sd = 2))
    data <- t(data)
    # 计算时间变化估计
    processed_data <- as.matrix(data)
    cot_results <- CoT(processed_data, true_expectations)
    
    # 输出结果
    output$cot_results <- renderPrint({
      cat("使用默认数据计算的时间变化估计结果：\n")
      cat("时间变化估计 CoT:\n")
      print(cot_results$CoT)
      cat("左置信区间 (lCoT):\n")
      print(cot_results$lCoT)
      cat("右置信区间 (rCoT):\n")
      print(cot_results$rCoT)
    })
    
    # 显示默认数据结构
    output$default_cot_data_display <- renderPrint({
      cat("默认数据示例（矩阵形式，真实被测量值为（10, 12, 11, 9, 13））：\n")
      print(processed_data)
    })
  })
}