library(ggplot2)

calculate_and_plot_FROC_with_AF <- function(t, L, X, n1, Y1, n0, Y0) {
  # 计算k1和k0
  k1 <- length(t)
  k0 <- length(n0)
  
  # 计算T
  T <- sum(t)
  
  # 计算LLF(ζ)和NLF(ζ)
  zeta_values <- sort(unique(c(unlist(X), unlist(Y1), unlist(Y0))))
  LLF <- numeric(length(zeta_values))
  NLF <- numeric(length(zeta_values))
  
  for (i in seq_along(zeta_values)) {
    zeta <- zeta_values[i]
    
    # 计算LLF(ζ)
    LLF[i] <- sum(sapply(1:k1, function(j) sum(L[[j]] == 1 & X[[j]] >= zeta))) / sum(t)
    
    # 计算NLF(ζ)
    NLF[i] <- (sum(sapply(1:k0, function(j) sum(Y0[[j]] >= zeta))) + 
                 sum(sapply(1:k1, function(j) sum(Y1[[j]] >= zeta)))) / (k1 + k0)
  }
  
  # 绘制FROC图像
  df_FROC <- data.frame(NLF = NLF, LLF = LLF)
  p <- ggplot(df_FROC, aes(x = NLF, y = LLF)) +
    geom_line() +
    geom_point() +
    labs(title = "FROC Curve", x = "NLF(ζ)", y = "LLF(ζ)") +
    theme_minimal()
  print(p)
  
  # 计算A_F
  psi <- function(x, y) {
    (x > y) + 0.5 * (x == y)
  }
  
  w1 <- matrix(0, k1, k1)
  w0 <- matrix(0, k0, k0)
  
  for (s_prime in 1:k1) {
    for (s in 1:k1) {
      if (sum(L[[s_prime]]) * n1[s] != 0) {
        w1[s_prime, s] <- sum(sapply(which(L[[s_prime]] == 1), function(i) {
          sum(sapply(1:n1[s], function(j) psi(X[[s_prime]][i], Y1[[s]][j])))
        }))
      }
    }
  }
  
  for (s_prime in 1:k0) {
    for (s in 1:k0) {
      if (sum(L[[s_prime]]) * n0[s] != 0) {
        w0[s_prime, s] <- sum(sapply(which(L[[s_prime]] == 1), function(i) {
          sum(sapply(1:n0[s], function(j) psi(X[[s_prime]][i], Y0[[s]][j])))
        }))
      }
    }
  }
  
  A_F <- (T * (k1 + k0))^(-1) * (sum(w1) + sum(w0))
  
  # 返回结果
  return(list(LLF = LLF, NLF = NLF, A_F = A_F))
}


library(dplyr)

# 数据预处理函数
preprocess_data <- function(data) {
  pos_data <- data %>% filter(Status == "Positive")
  neg_data <- data %>% filter(Status == "Negative")
  
  t <- pos_data$LesionCount
  L <- lapply(1:nrow(pos_data), function(i) {
    as.numeric(pos_data[i, 4])  # 取病灶标记
  })
  
  X <- lapply(1:nrow(pos_data), function(i) {
    if (pos_data$Marked1[i] == 1) {
      c(pos_data$Confidence1[i])
    } else {
      -Inf
    }
  })
  
  n1 <- pos_data$FalsePositivesCount
  
  Y1 <- lapply(1:nrow(pos_data), function(i) {
    c(pos_data$FPConfidence1[i])
  })
  
  n0 <- neg_data$FalsePositivesCount
  Y0 <- lapply(1:nrow(neg_data), function(i) {
    c(neg_data$FPConfidence1[i])
  })
  
  list(t = t, L = L, X = X, n1 = n1, Y1 = Y1, n0 = n0, Y0 = Y0)
}




calculate_and_plot_iROC_with_AiR <- function(t, L, X, n1, Y1, n0, Y0) {
  # 计算k1和k0
  k1 <- length(t)
  k0 <- length(n0)
  
  # 计算T
  T <- sum(t)
  
  # 计算ϕ1_i
  phi1 <- sapply(1:k1, function(i) ifelse(sum(L[[i]]) != 0 || n1[i] != 0, 1, 0))
  
  # 计算ϕ0_j
  phi0 <- sapply(1:k0, function(j) ifelse(n0[j] != 0, 1, 0))
  
  # 计算z1_i
  z1 <- sapply(1:k1, function(i) {
    if (sum(L[[i]]) != 0 || n1[i] != 0) {
      max(c(X[[i]][L[[i]] == 1], unlist(Y1[[i]])))
    } else {
      -Inf
    }
  })
  
  # 计算z0_j
  z0 <- sapply(1:k0, function(j) {
    if (n0[j] != 0) {
      max(Y0[[j]])
    } else {
      -Inf
    }
  })
  
  # 计算TPF_iR(ζ)和FPF_iR(ζ)
  zeta_values <- sort(unique(c(z1, z0)))
  TPF_iR <- numeric(length(zeta_values))
  FPF_iR <- numeric(length(zeta_values))
  
  for (i in seq_along(zeta_values)) {
    zeta <- zeta_values[i]
    
    # 计算TPF_iR(ζ)
    TPF_iR[i] <- sum(phi1 == 1 & z1 >= zeta) / k1
    
    # 计算FPF_iR(ζ)
    FPF_iR[i] <- sum(phi0 == 1 & z0 >= zeta) / k0
  }
  
  # 确保FPF_iR和TPF_iR的对应关系是单调增加的
  df_iROC <- data.frame(FPF_iR = FPF_iR, TPF_iR = TPF_iR)
  df_iROC <- df_iROC[order(df_iROC$FPF_iR, df_iROC$TPF_iR), ]
  
  # 绘制iROC曲线
  p <- ggplot(df_iROC, aes(x = FPF_iR, y = TPF_iR)) +
    geom_line() +
    geom_point() +
    annotate("segment", x = 0, y = 0, xend = min(df_iROC$FPF_iR), yend = 0, linetype = "dashed", color = "blue") +
    annotate("segment", x = max(df_iROC$FPF_iR), y = max(df_iROC$TPF_iR), xend = 1, yend = 1, linetype = "dashed", color = "red") +
    labs(title = "iROC Curve", x = "FPF_iR(ζ)", y = "TPF_iR(ζ)") +
    xlim(0, 1) +
    ylim(0, 1) +
    theme_minimal()
  print(p)
  
  # 计算A_iR
  psi <- function(x, y) {
    (x > y) + 0.5 * (x == y)
  }
  
  w <- matrix(0, k1, k0)
  
  for (s_prime in 1:k1) {
    for (s in 1:k0) {
      if (phi1[s_prime] == 1 && phi0[s] == 1) {
        w[s_prime, s] <- psi(z1[s_prime], z0[s])
      } else if (phi1[s_prime] == 1 && phi0[s] == 0) {
        w[s_prime, s] <- 1
      } else if (phi1[s_prime] == 0 && phi0[s] == 0) {
        w[s_prime, s] <- 0.5
      } else if (phi1[s_prime] == 0 && phi0[s] == 1) {
        w[s_prime, s] <- 0
      }
    }
  }
  
  A_iR <- sum(w) / (k1 * k0)
  
  # 返回结果
  return(list(TPF_iR = df_iROC$TPF_iR, FPF_iR = df_iROC$FPF_iR, A_iR = A_iR))
}










calculate_and_plot_AFROC_with_AAF <- function(t, L, X, n1, Y1, n0, Y0) {
  # 计算k1和k0
  k1 <- length(t)
  k0 <- length(n0)
  
  # 计算T
  T <- sum(t)
  
  # 计算ϕ0_j
  phi0 <- sapply(1:k0, function(j) ifelse(n0[j] != 0, 1, 0))
  
  # 计算z0_j
  z0 <- sapply(1:k0, function(j) {
    if (n0[j] != 0) {
      max(Y0[[j]])
    } else {
      -Inf
    }
  })
  
  # 计算LLFA(ζ)和FPFA(ζ)
  zeta_values <- sort(unique(c(unlist(X), z0)))
  LLFA <- numeric(length(zeta_values))
  FPFA <- numeric(length(zeta_values))
  
  for (i in seq_along(zeta_values)) {
    zeta <- zeta_values[i]
    
    # 计算LLFA(ζ)
    LLFA[i] <- sum(sapply(1:k1, function(j) sum(L[[j]] == 1 & X[[j]] >= zeta))) / sum(t)
    
    # 计算FPFA(ζ)
    FPFA[i] <- sum(phi0 == 1 & z0 >= zeta) / k0
  }
  
  # 确保FPFA和LLFA的对应关系是单调增加的
  df_AFROC <- data.frame(FPFA = FPFA, LLFA = LLFA)
  df_AFROC <- df_AFROC[order(df_AFROC$FPFA, df_AFROC$LLFA), ]
  
  # 绘制AFROC图像
  p <- ggplot(df_AFROC, aes(x = FPFA, y = LLFA)) +
    geom_line() +
    geom_point() +
    labs(title = "AFROC Curve", x = "FPFA(ζ)", y = "LLFA(ζ)") +
    xlim(0, 1) +
    ylim(0, 1) +
    theme_minimal()
  print(p)
  
  # 计算AAF
  psi <- function(x, y) {
    (x > y) + 0.5 * (x == y)
  }
  
  wA <- matrix(0, k1, k0)
  
  for (s_prime in 1:k1) {
    for (s in 1:k0) {
      if (sum(L[[s_prime]]) != 0 && phi0[s] == 1) {
        wA[s_prime, s] <- sum(sapply(which(L[[s_prime]] == 1), function(r) psi(X[[s_prime]][r], z0[s])))
      } else if (sum(L[[s_prime]]) != 0 && phi0[s] == 0) {
        wA[s_prime, s] <- (t[s_prime] - sum(L[[s_prime]])) / 2
      } else if (sum(L[[s_prime]]) == 0 && phi0[s] == 0) {
        wA[s_prime, s] <- t[s_prime] / 2
      } else if (sum(L[[s_prime]]) == 0 && phi0[s] == 1) {
        wA[s_prime, s] <- 0
      }
    }
  }
  
  AAF <- sum(wA) / (T * k0)
  
  # 返回结果
  return(list(LLFA = df_AFROC$LLFA, FPFA = df_AFROC$FPFA, AAF = AAF))
}


















#12.3.3 Resampling Methods
calculate_FAUC_with_SD <- function(t, L, X, n1, Y1, n0, Y0, p = NULL) {
  # 计算k1和k0
  k1 <- length(t)
  k0 <- length(n0)
  
  # 计算T
  T <- sum(t)
  
  # 计算ϕ1_i
  phi1 <- sapply(1:k1, function(i) ifelse(sum(L[[i]]) != 0 || n1[i] != 0, 1, 0))
  
  # 计算ϕ0_j
  phi0 <- sapply(1:k0, function(j) ifelse(n0[j] != 0, 1, 0))
  
  # 计算z1_i
  z1 <- sapply(1:k1, function(i) {
    if (!is.na(sum(L[[i]])) && sum(L[[i]]) != 0 || !is.na(n1[i]) && n1[i] != 0) {
      max(c(X[[i]][L[[i]] == 1], unlist(Y1[[i]])))
    } else {
      -Inf
    }
  })
  
  # 计算z0_j
  z0 <- sapply(1:k0, function(j) {
    if (!is.na(n0[j]) && n0[j] != 0) {
      max(Y0[[j]])
    } else {
      -Inf
    }
  })
  
  # 计算LLF(ζ)和NLF(ζ)
  zeta_values <- sort(unique(c(unlist(X), unlist(Y1), unlist(Y0))))
  LLF <- numeric(length(zeta_values))
  NLF <- numeric(length(zeta_values))
  
  for (i in seq_along(zeta_values)) {
    zeta <- zeta_values[i]
    
    # 计算LLF(ζ)
    LLF[i] <- sum(sapply(1:k1, function(j) sum(L[[j]] == 1 & X[[j]] >= zeta))) / sum(t)
    
    # 计算NLF(ζ)
    NLF[i] <- (sum(sapply(1:k0, function(j) sum(Y0[[j]] >= zeta))) + 
                 sum(sapply(1:k1, function(j) sum(Y1[[j]] >= zeta)))) / (k1 + k0)
  }
  
  # 确保NLF和LLF的对应关系是单调增加的
  df_FROC <- data.frame(NLF = NLF, LLF = LLF)
  df_FROC <- df_FROC[order(df_FROC$NLF, df_FROC$LLF), ]
  
  # 计算FAUC
  FAUC <- sum(diff(df_FROC$NLF) * (df_FROC$LLF[-1] + df_FROC$LLF[-nrow(df_FROC)]) / 2)
  
  # Jackknife方法计算FAUC的标准差
  jackknife_FAUC <- numeric(k1 + k0)
  
  for (i in 1:k1) {
    # 删除第i个阳性病例
    t_jack <- t[-i]
    L_jack <- L[-i]
    X_jack <- X[-i]
    n1_jack <- n1[-i]
    Y1_jack <- Y1[-i]
    
    jackknife_FAUC[i] <- calculate_FAUC(t_jack, L_jack, X_jack, n1_jack, Y1_jack, n0, Y0)
  }
  
  for (j in 1:k0) {
    # 删除第j个阴性病例
    n0_jack <- n0[-j]
    Y0_jack <- Y0[-j]
    
    jackknife_FAUC[k1 + j] <- calculate_FAUC(t, L, X, n1, Y1, n0_jack, Y0_jack)
  }
  
  jackknife_FAUC_mean <- mean(jackknife_FAUC)
  jackknife_FAUC_var <- ((k1 + k0 - 1) / (k1 + k0)) * sum((jackknife_FAUC - jackknife_FAUC_mean)^2)
  jackknife_FAUC_sd <- sqrt(jackknife_FAUC_var)
  
  # Bootstrap方法计算FAUC的标准差
  if (!is.null(p)) {
    bootstrap_FAUC <- numeric(p)
    
    for (b in 1:p) {
      # 从阳性样本中抽取大于一半的样本
      k1_boot <- sample(1:k1, ceiling(k1 / 2) + 1, replace = TRUE)
      t_boot <- t[k1_boot]
      L_boot <- L[k1_boot]
      X_boot <- X[k1_boot]
      n1_boot <- n1[k1_boot]
      Y1_boot <- Y1[k1_boot]
      
      # 从阴性样本中抽取大于一半的样本
      k0_boot <- sample(1:k0, ceiling(k0 / 2) + 1, replace = TRUE)
      n0_boot <- n0[k0_boot]
      Y0_boot <- Y0[k0_boot]
      
      bootstrap_FAUC[b] <- calculate_FAUC(t_boot, L_boot, X_boot, n1_boot, Y1_boot, n0_boot, Y0_boot)
    }
    
    bootstrap_FAUC_mean <- mean(bootstrap_FAUC)
    bootstrap_FAUC_var <- var(bootstrap_FAUC)
    bootstrap_FAUC_sd <- sqrt(bootstrap_FAUC_var)
  } else {
    bootstrap_FAUC_sd <- NA
  }
  
  # 返回结果
  return(list(FAUC = FAUC, 
              jackknife_FAUC_sd = jackknife_FAUC_sd, bootstrap_FAUC_sd = bootstrap_FAUC_sd))
}

# 辅助函数：计算FAUC
calculate_FAUC <- function(t, L, X, n1, Y1, n0, Y0) {
  k1 <- length(t)
  k0 <- length(n0)
  T <- sum(t)
  
  phi1 <- sapply(1:k1, function(i) ifelse(sum(L[[i]]) != 0 || n1[i] != 0, 1, 0))
  phi0 <- sapply(1:k0, function(j) ifelse(n0[j] != 0, 1, 0))
  
  z1 <- sapply(1:k1, function(i) {
    if (!is.na(sum(L[[i]])) && sum(L[[i]]) != 0 || !is.na(n1[i]) && n1[i] != 0) {
      max(c(X[[i]][L[[i]] == 1], unlist(Y1[[i]])))
    } else {
      -Inf
    }
  })
  
  z0 <- sapply(1:k0, function(j) {
    if (!is.na(n0[j]) && n0[j] != 0) {
      max(Y0[[j]])
    } else {
      -Inf
    }
  })
  
  zeta_values <- sort(unique(c(unlist(X), unlist(Y1), unlist(Y0))))
  LLF <- numeric(length(zeta_values))
  NLF <- numeric(length(zeta_values))
  
  for (i in seq_along(zeta_values)) {
    zeta <- zeta_values[i]
    
    LLF[i] <- sum(sapply(1:k1, function(j) sum(L[[j]] == 1 & X[[j]] >= zeta))) / sum(t)
    NLF[i] <- (sum(sapply(1:k0, function(j) sum(Y0[[j]] >= zeta))) + 
                 sum(sapply(1:k1, function(j) sum(Y1[[j]] >= zeta)))) / (k1 + k0)
  }
  
  df_FROC <- data.frame(NLF = NLF, LLF = LLF)
  df_FROC <- df_FROC[order(df_FROC$NLF, df_FROC$LLF), ]
  
  FAUC <- sum(diff(df_FROC$NLF) * (df_FROC$LLF[-1] + df_FROC$LLF[-nrow(df_FROC)]) / 2)
  
  return(FAUC)
}





