library(shiny)
library(ggplot2)
library(pracma)

# ALROC 计算函数
ALROC <- function(X, Z, c) {
  k0 <- length(X)
  k1 <- nrow(Z)
  TLR <- sum(Z[, 1] > c & Z[, 2] == 1) / k1
  FPF <- sum(X > c) / k0
  ALROC <- 0
  for (j in 1:k1) {
    for (i in 1:k0) {
      Cj <- Z[j, 2]
      Yj <- Z[j, 1]
      Xi <- X[i]
      Psi <- (Yj > Xi) + 0.5 * (Yj == Xi)
      ALROC <- ALROC + Cj * Psi
    }
  }
  ALROC <- ALROC / (k1 * k0)
  return(list(TLR = TLR, FPF = FPF, ALROC = ALROC))
}

# 计算 ALROC 和 σL
nonparametricALROC_sigmaL <- function(X, Z, c) {
  k0 <- length(X)
  k1 <- nrow(Z)
  
  phi <- function(t) dnorm(t)
  Phi <- function(t) pnorm(t)
  
  H_epsilon <- function(y) {
    integrate(function(x) phi(x) * Phi(x + y), -Inf, Inf)$value
  }
  
  ALROC <- 0
  for (j in 1:k1) {
    for (i in 1:k0) {
      Cj <- Z[j, 2]
      Yj <- Z[j, 1]
      Xi <- X[i]
      ALROC <- ALROC + Cj * H_epsilon(Yj - Xi)
    }
  }
  ALROC <- ALROC / (k0 * k1)
  
  Q <- sum(Z[, 2]) / k1
  
  A_c <- 0
  for (j in 1:k1) {
    for (i in 1:k0) {
      Cj <- Z[j, 2]
      Yj <- Z[j, 1]
      Xi <- X[i]
      A_c <- A_c + Cj * H_epsilon(Yj - Xi)
    }
  }
  A_c <- A_c / (k0 * k1)
  
  B_c11 <- 0
  for (j in 1:k1) {
    for (i in 1:k0) {
      Cj <- Z[j, 2]
      Yj <- Z[j, 1]
      Xi <- X[i]
      B_c11 <- B_c11 + Cj * H_epsilon(Yj - Xi)^2
    }
  }
  B_c11 <- B_c11 / (k0 * k1)
  
  B_c12 <- 0
  for (j in 1:k1) {
    for (i in 1:k0) {
      for (i_prime in 1:k0) {
        Cj <- Z[j, 2]
        Yj <- Z[j, 1]
        Xi <- X[i]
        Xi_prime <- X[i_prime]
        B_c12 <- B_c12 + Cj * H_epsilon(Yj - Xi) * H_epsilon(Yj - Xi_prime)
      }
    }
  }
  B_c12 <- B_c12 / (k1 * k0^2)
  
  B_c21 <- 0
  for (j in 1:k1) {
    for (i in 1:k0) {
      for (j_prime in 1:k1) {
        Cj <- Z[j, 2]
        Yj <- Z[j, 1]
        Xi <- X[i]
        Yj_prime <- Z[j_prime, 1]
        B_c21 <- B_c21 + Cj * H_epsilon(Yj - Xi) * H_epsilon(Yj_prime - Xi)
      }
    }
  }
  B_c21 <- B_c21 / (k1^2 * k0)
  
  S_c0 <- B_c11 - (1 / Q) * A_c^2
  S_c1 <- B_c12 - (1 / Q) * A_c^2
  S_c2 <- B_c21 - A_c^2
  S_c3 <- (1 - Q) / Q * A_c^2
  
  sigmaL <- sqrt((S_c0 + (k0 - 1) * S_c1 + (k1 - 1) * S_c2 + k0 * S_c3) / (k0 * k1))
  
  return(list(ALROC = ALROC, sigmaL = sigmaL))
}


# 第一个子页面的Server
server1 <- function(input, output, session) {
  observeEvent(input$run, {
    if (!is.null(input$X_input) && input$X_input != "" &&
        !is.null(input$Y_input) && input$Y_input != "" &&
        !is.null(input$C_input) && input$C_input != "" &&
        !is.null(input$c_input) && input$c_input != "") {
      data <- parse_text_input(input$X_input, input$Y_input, input$C_input, input$c_input)
    } else if (!is.null(input$csv_input)) {
      data <- process_csv_input(input$csv_input)
    } else if (!is.null(input$excel_input)) {
      data <- process_excel_input(input$excel_input)
    } else {
      X <- c(1, 2, 3, 4, 5)
      Y <- c(2, 3, 4, 5)
      C <- c(1, 1, 0, 1)
      c <- 3
      Z <- cbind(Y, C)
      data <- list(X = X, Z = Z, c = c)
    }
    
    result <- ALROC(data$X, data$Z, data$c)
    
    # 输出格式: 变量名：数值
    output$results <- renderPrint({
      cat("TLR:", result$TLR, "\n")
      cat("FPF:", result$FPF, "\n")
      cat("ALROC:", result$ALROC, "\n")
    })
  })
}




# 第二个子页面的Server
server2 <- function(input, output, session) {
  nonparametricTLR_FPF_PLOT <- function(X, Z, c) {
    k0 <- length(X)
    k1 <- nrow(Z)
    
    phi <- function(t) dnorm(t)
    Phi <- function(t) pnorm(t)
    
    zeta_values <- seq(min(c(X, Z[, 1])) - 3, max(c(X, Z[, 1])) + 3, length.out = 1000)
    f_c <- sapply(zeta_values, function(zeta) {
      sum(Z[, 2] * phi(zeta - Z[, 1])) / k1
    })
    g <- sapply(zeta_values, function(zeta) {
      sum(phi(zeta - X)) / k0
    })
    
    df_fg <- data.frame(zeta = zeta_values, f_c = f_c, g = g)
    
    output$density_plot_np <- renderPlot({
      ggplot(df_fg, aes(x = zeta)) +
        geom_line(aes(y = f_c, color = "f_c(ζ)")) +
        geom_line(aes(y = g, color = "g(ζ)")) +
        labs(title = "f_c(ζ) and g(ζ)", x = "ζ", y = "Density") +
        scale_color_manual(values = c("f_c(ζ)" = "blue", "g(ζ)" = "red")) +
        theme_minimal()
    })
    
    F_c <- sapply(zeta_values, function(zeta) {
      sum(Z[, 2] * Phi(zeta - Z[, 1])) / k1
    })
    G <- sapply(zeta_values, function(zeta) {
      sum(Phi(zeta - X)) / k0
    })
    
    df_FG <- data.frame(zeta = zeta_values, F_c = F_c, G = G)
    
    output$cumulative_plot_np <- renderPlot({
      ggplot(df_FG, aes(x = zeta)) +
        geom_line(aes(y = F_c, color = "F_c(ζ)")) +
        geom_line(aes(y = G, color = "G(ζ)")) +
        labs(title = "F_c(ζ) and G(ζ)", x = "ζ", y = "Cumulative Probability") +
        scale_color_manual(values = c("F_c(ζ)" = "blue", "G(ζ)" = "red")) +
        theme_minimal()
    })
    
    TLF_c <- integrate(function(y) sapply(y, function(zeta) sum(Z[, 2] * phi(zeta - Z[, 1])) / k1), lower = c, upper = Inf)$value
    FPF_c <- 1 - sum(Phi(c - X)) / k0
    
    ALROC <- 0
    for (j in 1:k1) {
      for (i in 1:k0) {
        Cj <- Z[j, 2]
        Yj <- Z[j, 1]
        Xi <- X[i]
        Psi <- (Yj > Xi) + 0.5 * (Yj == Xi)
        ALROC <- ALROC + Cj * Psi
      }
    }
    ALROC <- ALROC / (k1 * k0)
    
    return(list(TLF_c = TLF_c, FPF_c = FPF_c, ALROC = ALROC))
  }
  
  observeEvent(input$run_np, {
    if (!is.null(input$X_input_np) && input$X_input_np != "" &&
        !is.null(input$Y_input_np) && input$Y_input_np != "" &&
        !is.null(input$C_input_np) && input$C_input_np != "" &&
        !is.null(input$c_input_np) && input$c_input_np != "") {
      data <- parse_text_input(input$X_input_np, input$Y_input_np, input$C_input_np, input$c_input_np)
    } else if (!is.null(input$csv_input_np)) {
      data <- process_csv_input(input$csv_input_np)
    } else if (!is.null(input$excel_input_np)) {
      data <- process_excel_input(input$excel_input_np)
    } else {
      X <- c(1, 2, 3, 4, 5)
      Z <- matrix(c(2, 1, 3, 1, 4, 0, 5, 1), nrow = 4, byrow = TRUE)
      c <- 3
      data <- list(X = X, Z = Z, c = c)
    }
    
    result <- nonparametricTLR_FPF_PLOT(data$X, data$Z, data$c)
    output$results_np <- renderPrint({
      cat("TLF_c:", result$TLF_c, "\n")
      cat("FPF_c:", result$FPF_c, "\n")
      cat("ALROC:", result$ALROC, "\n")
    })
  })
}



# 第三个子页面的Server
server3 <- function(input, output, session) {
  observeEvent(input$run_alroc, {
    if (!is.null(input$X_input_alroc) && input$X_input_alroc != "" &&
        !is.null(input$Y_input_alroc) && input$Y_input_alroc != "" &&
        !is.null(input$C_input_alroc) && input$C_input_alroc != "" &&
        !is.null(input$c_input_alroc) && input$c_input_alroc != "") {
      data <- parse_text_input(input$X_input_alroc, input$Y_input_alroc, input$C_input_alroc, input$c_input_alroc)
    } else if (!is.null(input$csv_input_alroc)) {
      data <- process_csv_input(input$csv_input_alroc)
    } else if (!is.null(input$excel_input_alroc)) {
      data <- process_excel_input(input$excel_input_alroc)
    } else {
      X <- c(1, 2, 3, 4, 5)
      Z <- matrix(c(2, 1, 3, 1, 4, 0, 5, 1), nrow = 4, byrow = TRUE)
      c <- 3
      data <- list(X = X, Z = Z, c = c)
    }
    
    result <- nonparametricALROC_sigmaL(data$X, data$Z, data$c)
    
    # 只输出 ALROC 和 σL
    output$results_alroc <- renderPrint({
      cat("ALROC:", result$ALROC, "\n")
      cat("σL:", result$sigmaL, "\n")
    })
  })
}

