library(shiny)
library(readr)
library(readxl)
library(ggplot2)

# 定义计算和绘图的函数
calculate_and_plot_ROI <- function(y, x) {
  S <- length(y)
  m <- sapply(y, length)
  n <- sapply(x, length)
  M <- sum(m)
  N <- sum(n)
  
  zeta_values <- sort(unique(c(unlist(y), unlist(x))))
  TPF <- numeric(length(zeta_values))
  FPF <- numeric(length(zeta_values))
  
  for (i in seq_along(zeta_values)) {
    zeta <- zeta_values[i]
    TPF[i] <- sum(sapply(1:S, function(j) sum(y[[j]] > zeta))) / M
    FPF[i] <- sum(sapply(1:S, function(j) sum(x[[j]] > zeta))) / N
  }
  
  df_ROI <- data.frame(FPF = FPF, TPF = TPF)
  df_ROI <- df_ROI[order(df_ROI$FPF, df_ROI$TPF), ]
  
  p <- ggplot(df_ROI, aes(x = FPF, y = TPF)) +
    geom_line() +
    geom_point() +
    labs(title = "ROI Curve", x = "FPF(ζ)", y = "TPF(ζ)") +
    xlim(0, 1) +
    ylim(0, 1) +
    theme_minimal()
  
  print(p)
  
  psi <- function(x, y) {
    (x > y) + 0.5 * (x == y)
  }
  
  AROI <- 0
  for (i in 1:S) {
    for (j in 1:m[i]) {
      for (s in 1:S) {
        for (k in 1:n[s]) {
          AROI <- AROI + psi(y[[i]][j], x[[s]][k])
        }
      }
    }
  }
  
  AROI <- AROI / (M * N)
  
  return(list(TPF = df_ROI$TPF, FPF = df_ROI$FPF, AROI = AROI))
}

# 定义默认数据
default_data <- function() {
  y <- list(
    c(0.8, 0.9, 0.7), 
    c(0.7, 0.8, 0.9), 
    c(0.6, 0.8, 0.9, 0.7)  
  )
  x <- list(
    c(0.3, 0.4),  
    c(0.4, 0.5, 0.6),  
    c(0.3, 0.4, 0.5, 0.6)  
  )
  return(list(y = y, x = x))
}

# Server
server <- function(input, output, session) {
  
  # 反应值存储数据
  current_data <- reactiveVal()
  
  observeEvent(input$data_file, {
    req(input$data_file)
    if (grepl("\\.csv$", input$data_file$name)) {
      data <- read_csv(input$data_file$datapath)
      y <- split(data$y, seq(nrow(data)))
      x <- split(data$x, seq(nrow(data)))
      current_data(list(y = y, x = x))
    } else if (grepl("\\.xlsx$", input$data_file$name)) {
      data <- read_excel(input$data_file$datapath)
      y <- split(data$y, seq(nrow(data)))
      x <- split(data$x, seq(nrow(data)))
      current_data(list(y = y, x = x))
    }
  })
  
  observeEvent(input$use_default, {
    data <- default_data()
    current_data(data)
    
    output$results <- renderPrint({
      cat("默认数据:\n")
      cat("y:", unlist(data$y), "\n")
      cat("x:", unlist(data$x), "\n")
      result <- calculate_and_plot_ROI(data$y, data$x)
      cat("AROI:", result$AROI, "\n")
    })
    
    output$roi_plot <- renderPlot({
      calculate_and_plot_ROI(data$y, data$x)
    })
  })
  
  observeEvent(input$run_analysis, {
    if (is.null(current_data())) {
      data <- default_data()
    } else {
      data <- current_data()
    }
    
    output$roi_plot <- renderPlot({
      calculate_and_plot_ROI(data$y, data$x)
    })
    
    output$results <- renderPrint({
      result <- calculate_and_plot_ROI(data$y, data$x)
      cat("AROI:", result$AROI, "\n")
    })
  })
}