#' Title 11.5em
#'
#' @param N The total number of people in this crowd
#' @param t An N*I dimensional matrix representing the observed data
#' @param I Total number of detections
#' @param product A vector of 2*I+1 dimensions, with the first dimension representing prevalence, the next I dimension representing sensitivity, and the last I dimension representing specificity
#'
#' @return The results of the next iteration
#' @export
#'
#'
estimation_method2 <- function(N,t,I,product){
  pi=product[1]
  Se=vector()
  Sp=vector()
  A1=vector()
  B1=vector()
  q1=vector()
  for(h in 1:I){
    Se[h]=product[h+1]
    Sp[h]=product[h+I+1]
  }
  for(k in 1:N){
    A=rep(1,N)
    B=rep(1,N)
    q=rep(1,N)
    for(i in 1:I){
      A[k]=A[k]*(Se[i]^t[i,k])*((1-Se[i])^(1-t[i,k]))
      B[k]=B[k]*((1-Sp[i])^t[i,k])*(Sp[i]^(1-t[i,k]))
    }
    A[k]=A[k]*pi
    B[k]=B[k]*(1-pi)
    q[k]=A[k]/(A[k]+B[k])
    A1[k]=A[k]
    B1[k]=B[k]
    q1[k]=q[k]
  }
  pi=sum(q1)/N
  for(i in 1:I){
    Se[i]=sum(t[i,]*q1)/sum(q1)
    Sp[i]=sum((1-t[i,])*(1-q1))/sum(1-q1)
  }
  product=c(pi,Se,Sp)
  return(product)
}

#' Title 11.5em(cia)
#'
#' @param init_pi Initial value of prevalence
#' @param N The total number of people in this crowd
#' @param input_t An N*I dimensional matrix representing the observed data
#' @param I Total number of detections
#' @param init_Se An I-dimensional vector representing the initial value of the sensitivity
#' @param init_Sp An I-dimensional vector representing the initial value of the specificity
#' @param n_iter Number of iterations
#' @param burn_in burn_in period
#'
#' @return Means and 95% confidence intervals for prevalence, sensitivity and specificity
#' @export
#'
#' @examples
#' my_input_data <- t(cbind(c(rep(0,1604), rep(1,88)),c(rep(0,1534), rep(1,70), rep(0,42), rep(1,46)),c(rep(0,1513), rep(1,21), rep(0,59), rep(1,11),rep(0,23), rep(1,19), rep(0,12), rep(1,34))))
#' em_algorithm2(init_pi = 0.8, N = 1692, input_t = my_input_data, I = 3, init_Se = c(0.855, 0.750, 0.842), init_Sp = c(0.986, 0.963, 0.987), n_iter = 500, burn_in = 0)
em_algorithm2 <- function(init_pi,N,input_t,I,init_Se,init_Sp,n_iter,burn_in){
  set.seed(1234)
  n <- N
  n_param <-2*I+1
  i <- 0
  c <- vector()
  trace <- matrix(0, n_iter-burn_in+1, n_param)
  c <- character(2*I+1)
  c[1] <- 'pi'
  for(i in 1:I) {
    c[i+1] <- paste("Se[", i, "]", sep="")
    c[i+I+1] <- paste("Sp[", i, "]", sep="")
  }
  colnames(trace) <- c
  index <- 1
  pi <- init_pi
  Se <- init_Se
  Sp <- init_Sp
  product <- matrix(0,nrow=n_iter,ncol=2*I+1)
  product[1,] <- c(pi,Se,Sp)
  for (i in 2:n_iter){
    product[i,]=estimation_method2(N,input_t,I,product[i-1,])
    if (i > burn_in) {
      trace[index, ] <- product[i,]
      index <- index + 1
    }
  }
  result <- list()
  for(j in 1:n_param){
    result[[colnames(trace)[j]]] <- c(median(trace[, j]),
                                      quantile(trace[, j], c(0.025, 0.975),na.rm=TRUE))
  }
  return(result)
}
