#' Title 11.3（CIA）
#'
#' @param inits A five-dimensional vector representing the prevalence, sensitivity and specificity of the initial
#' @param data Observations, a four-dimensional vector
#' @param n_iter Number of iterations
#' @param burn_in burn_in period
#' @param alpha.pi Prior information on prevalence
#' @param beta.pi Prior information on prevalence
#' @param alpha.sens1 Prior information on sensitivity
#' @param beta.sens1 Prior information on sensitivity
#' @param alpha.spec1 Prior information on specificity
#' @param beta.spec1 Prior information on specificity
#' @param alpha.sens2 Prior information on sensitivity
#' @param beta.sens2 Prior information on sensitivity
#' @param alpha.spec2 Prior information on specificity
#' @param beta.spec2 Prior information on specificity
#'
#' @return Means and 95% confidence intervals for prevalence, sensitivity and specificity
#' @export
#'
#' @examples gibbs2sampler(inits=c(0.5, 0.95, 0.9, 0.8, 0.9), data = c(38, 2, 87, 35), n_iter=20000, burn_in = 5000,alpha.pi = 1, beta.pi = 1, alpha.sens1 = 4.44, beta.sens1 = 13.31, alpha.spec1 = 71.25,beta.spec1 = 3.75, alpha.sens2 = 21.96, beta.sens2 = 5.49,alpha.spec2 = 4.1, beta.spec2 = 1.76)
gibbs2sampler <- function(inits, data, n_iter, burn_in, alpha.pi = 1, beta.pi = 1,
                          alpha.sens1 = 4.44, beta.sens1 = 13.31, alpha.spec1 = 71.25,
                          beta.spec1 = 3.75, alpha.sens2 = 21.96, beta.sens2 = 5.49,
                          alpha.spec2 = 4.1, beta.spec2 = 1.76){
  set.seed(123)
  n <- sum(data)
  n_param <- 5
  trace <- matrix(0, n_iter-burn_in+1, n_param)
  colnames(trace) <- c("pi", "sens1", "spec1", "sens2", "spec2")
  index <- 1
  pi <- inits[1]
  sens1 <- inits[2]
  spec1 <- inits[3]
  sens2 <- inits[4]
  spec2 <- inits[5]
  for(i in 1:n_iter){
    Y11 <- rbinom(data[1], 1, (pi*sens1*sens2)/(pi*sens1*sens2+(1-pi)*(1-spec1)*(1-spec2)))
    Y10 <- rbinom(data[2], 1, (pi*sens1*(1-sens2))/(pi*sens1*(1-sens2)+(1-pi)*(1-spec1)*spec2))
    Y01 <- rbinom(data[3], 1, (pi*(1-sens1)*sens2)/(pi*(1-sens1)*sens2+(1-pi)*spec1*(1-spec2)))
    Y00 <- rbinom(data[4], 1, (pi*(1-sens1)*(1-sens2))/(pi*(1-sens1)*(1-sens2)+(1-pi)*spec1*spec2))
    pi <- rbeta(1, sum(Y11, Y10, Y01, Y00)+alpha.pi, n-sum(Y11, Y10, Y01, Y00)+beta.pi)
    sens1 <- rbeta(1, sum(Y11, Y10)+alpha.sens1, sum(Y01, Y00)+beta.sens1)
    spec1 <- rbeta(1, sum(data[c(3, 4)])-sum(Y01, Y00)+alpha.spec1,
                   sum(data[c(1, 2)])-sum(Y11, Y10)+beta.spec1)
    sens2 <- rbeta(1, sum(Y11, Y01)+alpha.sens2, sum(Y10, Y00)+beta.sens2)
    spec2 <- rbeta(1, sum(data[c(2, 4)])-sum(Y10, Y00)+alpha.spec2,
                   sum(data[c(1, 3)])-sum(Y11, Y01)+beta.spec2)

    if(i > burn_in){
      trace[index, ] <- c(pi, sens1, spec1, sens2, spec2)
      index <- index+1
    }
  }
  result <- list()
  for(j in 1:n_param){
    result[[colnames(trace)[j]]] <- c(median(trace[, j]), quantile(trace[, j], c(0.025, 0.975), na.rm=T))
  }
  return(result)
}
