#' Title 11.4em
#'
#' @param N A G-dimensional vector, where each term represents the total number of people in each population
#' @param t A Ng*G-dimensional matrix representing the observed data of the first test
#' @param r A Ng*G-dimensional matrix representing the observed data of the second test
#' @param G Number of people
#' @param product A G+4-dimensional vector representing the results of the last iteration, with the former G dimension representing the prevalence and the latter representing the sensitivity and specificity of the two tests
#'
#' @return The results of the next iteration
#' @export
#'
#'
estimation_method <- function(N,t,r,G,product){
  n=sum(N)
  pi=product[1:G]
  Se1=product[G+1]
  Sp1=product[G+2]
  Se2=product[G+3]
  Sp2=product[G+4]
  for(g in 1:G){
    A=matrix(0,nrow=N[g],ncol=G)
    B=matrix(0,nrow=N[g],ncol=G)
    q=matrix(0,nrow=N[g],ncol=G)
    for(k in 1:N[g]){
      A[k,g]=(Se1)^(t[k,g])*(1-Se1)^(1-t[k,g])*
        (Se2)^(r[k,g])*(1-Se2)^(1-r[k,g])*pi[g]
      B[k,g]=(1-Sp1)^(t[k,g])*(Sp1)^(1-t[k,g])*
        (1-Sp2)^(r[k,g])*(Sp2)^(1-r[k,g])*(1-pi[g])
      q[k,g]=A[k,g]/(A[k,g]+B[k,g])
    }
    epsilon=N[g]/n
    pi[g]=sum(q[,g])/N[g]
  }
  Se1=(sum(t*q))/sum(q)
  Se2=(sum(r*q))/sum(q)
  Sp1=(sum((1-t)*(1-q)))/sum(1-q)
  Sp2=(sum((1-r)*(1-q)))/sum(1-q)
  product=c(pi,Se1,Sp1,Se2,Sp2)
  return(product)
}

#' Title 11.4em(cia)
#'
#' @param init_pi A G-dimensional vector representing the initial value of prevalence for each population
#' @param N A G-dimensional vector, where each term represents the total number of people in each population
#' @param t A Ng*G-dimensional matrix representing the observed data of the first test
#' @param r A Ng*G-dimensional matrix representing the observed data of the second test
#' @param G Number of people
#' @param init_Se1 Initial value of sensitivity
#' @param init_Sp1 Initial value of specificity
#' @param init_Se2 Initial value of sensitivity
#' @param init_Sp2 Initial value of specificity
#' @param n_iter Number of iterations
#' @param burn_in burn_in period
#'
#' @return Means and 95% confidence intervals for prevalence, sensitivity and specificity
#' @export
#'
#' @examples em_algorithm(init_pi=c(0.5,0.5),N=c(555,1322),t=cbind(c(rep(1,18),rep(0,537),rep(0,767)),c(rep(1,918), rep(0,404))),r=cbind(c(rep(1,23),rep(0,532),rep(0,767)),c(rep(1,924),rep(0,398))),G=2,init_Se1=0.8,init_Sp1=0.8,init_Se2=0.76,init_Sp2=0.64,n_iter=100,burn_in=20)
em_algorithm <- function(init_pi,N,t,r,G,init_Se1,init_Sp1,init_Se2,init_Sp2,n_iter,burn_in){
  n <- sum(N)
  n_param <- G+4
  i <- 0
  c <- vector()
  trace <- matrix(0, n_iter-burn_in+1, n_param)
  for(i in 1:G){c[i]=i}
  c[G+1] <- 'Se1'
  c[G+2] <- 'Sp1'
  c[G+3] <- 'Se2'
  c[G+4] <- 'Sp2'
  colnames(trace) <- c
  index <- 1
  pi <- init_pi
  Se1 <- init_Se1
  Sp1 <- init_Sp1
  Se2 <- init_Se2
  Sp2 <- init_Sp2
  product <- matrix(0,nrow=n_iter,ncol=G+4)
  product[1,] <- c(pi,Se1,Sp1,Se2,Sp2)
  for (i in 2:n_iter){
    product[i,]=estimation_method(N,t,r,G,product[i-1,])
    if((i > burn_in)&&(product[i,G+1]+product[i,G+2]>1)&&(product[i,G+3]+product[i,G+4]>1)){
      trace[index, ] <- product[i,]
      index <- index+1
    }
  }
  result <- list()
  for(j in 1:n_param){
    result[[colnames(trace)[j]]] <- c(median(trace[, j]),
                                      quantile(trace[, j], c(0.025, 0.975),na.rm=TRUE))
  }
  return(result)
}
