

#' Correction Methods Under the MAR Assumption with a Single Binary-Scale Test
#'
#' @param s0  number of subjects verified with the disease conditions and negative test result
#' @param r0  number of subjects verified without the disease conditions and negative test result
#' @param m0  number of subjects with negative test result
#' @param s1  number of subjects verified with the disease conditions and positive test result
#' @param r1  number of subjects verified without the disease conditions and positive test result
#' @param m1  number of subjects verified with positive test result
#' @param delta delta
#'
#' @return Se_hat, Sp_hat, Var_Se, Var_Sp
#' @importFrom stats qnorm
#' @export
#'
Correct_Bias_MAR1 <- function(s0, r0, m0, s1, r1, m1, delta = 0.95){
  N <- m0 + m1
  ##计算Se和Sp的矫正值及其方差
  Se_hat <- ((m1 * s1)/(s1 + r1))/((m0 * s0)/(s0 + r0) + (m1 * s1)/(s1 + r1))
  Sp_hat <- ((m0 * r0)/(s0 + r0))/((m0 * r0)/(s0 + r0) + (m1 * r1)/(s1 + r1))
  Var_Se_hat <- ((Se_hat * (1 - Se_hat))^2) * (N / (m0 * m1) + r1 / (s1*(s1 + r1)) + r0 / (s0*(s0 + r0)))
  Var_Sp_hat <- ((Sp_hat * (1 - Sp_hat))^2) * (N / (m0 * m1) + s1 / (r1*(s1 + r1)) + s0 / (r0*(s0 + r0)))

  ##用两种方法计算(1-delta)置信区间
  z <- qnorm(1 - delta / 2, mean = 0, sd = 1)

  ##通过渐近正态性直接计算置信区间
  lower_bound_Se_hat_1 <- Se_hat - z * sqrt(Var_Se_hat)
  upper_bound_Se_hat_1 <- Se_hat + z * sqrt(Var_Se_hat)
  lower_bound_Sp_hat_1 <- Sp_hat - z * sqrt(Var_Sp_hat)
  upper_bound_Sp_hat_1 <- Sp_hat + z * sqrt(Var_Sp_hat)


  ##通过计算Se和Sp的logit转换对应的置信区间获得Se和Sp的置信区间
  logit_Se_hat <- log(Se_hat / (1 - Se_hat))
  logit_Sp_hat <- log(Sp_hat / (1 - Sp_hat))
  Var_logit_Se_hat <- N / (m0 * m1) + r1 / (s1*(s1 + r1)) + r0 / (s0*(s0 + r0))
  Var_logit_Sp_hat <- N / (m0 * m1) + s1 / (r1*(s1 + r1)) + s0 / (r0*(s0 + r0))
  lower_bound_logit_Se_hat <- logit_Se_hat - z * sqrt(Var_logit_Se_hat)
  upper_bound_logit_Se_hat <- logit_Se_hat + z * sqrt(Var_logit_Se_hat)
  lower_bound_logit_Sp_hat <- logit_Sp_hat - z * sqrt(Var_logit_Sp_hat)
  upper_bound_logit_Sp_hat <- logit_Sp_hat + z * sqrt(Var_logit_Sp_hat)

  lower_bound_Se_hat_2 <- exp(lower_bound_logit_Se_hat) / (1 + exp(lower_bound_logit_Se_hat))
  upper_bound_Se_hat_2 <- exp(upper_bound_logit_Se_hat) / (1 + exp(upper_bound_logit_Se_hat))
  lower_bound_Sp_hat_2 <- exp(lower_bound_logit_Sp_hat) / (1 + exp(lower_bound_logit_Sp_hat))
  upper_bound_Sp_hat_2 <- exp(upper_bound_logit_Sp_hat) / (1 + exp(upper_bound_logit_Sp_hat))



  return <- list(Se_hat, Sp_hat, Var_Se_hat, Var_Sp_hat)
}

